/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import lombok.Getter;
import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.engine.typexp.AST;
import nz.org.riskscape.engine.typexp.AST.ComplexType;
import nz.org.riskscape.engine.typexp.ComplexTypeConstructor;
import nz.org.riskscape.engine.typexp.TypeBuilder;
import nz.org.riskscape.rl.ast.Lambda;

/**
 * Represents a lambda expression in an unrealized function's type signature.
 */
public class LambdaType implements Type {

  public static final ComplexTypeConstructor TYPE_CONSTRUCTOR = (TypeBuilder typeBuilder, ComplexType type) -> {
    List<AST> args = type.args();

    List<String> checkedArgs =
        typeBuilder.expectConstantsOfType(type, Types.TEXT, args, 0)
        .stream()
        .map(Object::toString)
        .collect(Collectors.toList());

    return new LambdaType(checkedArgs);
  };

  public static final TypeInformation TYPE_INFORMATION = new TypeInformation(
      "lambda",
      LambdaType.class,
      Lambda.class,
      TYPE_CONSTRUCTOR
  );

  public static final LambdaType NO_ARGS = new LambdaType(Collections.emptyList());

  /**
   * Create a type that matches a {@link Lambda} expression.
   */
  public static LambdaType create(Lambda lambda) {
    return new LambdaType(lambda.getArguments().stream().map(Token::getValue).collect(Collectors.toList()));
  }

  /**
   * The number of declared arguments
   */
  @Getter
  private final List<String> args;

  private final Struct asStruct;

  public LambdaType(List<String> argIdentifiers) {
    this.args = argIdentifiers;
    this.asStruct = Struct.of(SimpleType.DEFAULT_MEMBER_NAME, this);
  }

  public LambdaType(String... argIdentifiers) {
    this(Arrays.asList(argIdentifiers));
  }

  /**
   * @return a scoped version of a {@link LambdaType} that remembers the arity and the scope that was closed over when
   * the expression was realized
   */
  public ScopedLambdaType scoped(Struct scopeType) {
    return new ScopedLambdaType(scopeType, args);
  }

  @Override
  public Object coerce(Object value) throws CoercionException {
    throw new UnsupportedOperationException();
  }

  @Override
  public Class<?> internalType() {
    return Lambda.class;
  }

  @Override
  public Struct asStruct() {
    return asStruct;
  }

  @Override
  public int estimateSize(Object entry) {
    return -1;
  }

  @Override
  public void toBytes(DataOutputStream os, Object toWrite) throws IOException {
    throw new UnsupportedOperationException();
  }

  @Override
  public Object fromBytes(DataInputStream in) throws IOException {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean equals(Object obj) {
    if (obj instanceof LambdaType) {
      // NB I'm not 100% on this - it might be enough to base equality on the number of args, or maybe the names
      // matter - not sure yet...
      return ((LambdaType) obj).args.size() == this.args.size();
    } else {
      return false;
    }
  }

  @Override
  public String toString() {
    return String.format("λ(%s)", args.stream().collect(Collectors.joining(",")));
  }

  @Override
  public int hashCode() {
    // lazy, but awesome
    return Objects.hashCode(args);
  }

  @Override
  public <T, U> U visit(TypeVisitor<T, U> tv, T data) {
    return tv.atomicType(this, data);
  }


}
