/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;

/**
 *
* Permissive Integer type that will convert any {@link java.lang.Number} in to
* an long integer via {@link Number#longValue()}. It will also attempt to convert a
* {@link java.lang.String} to an integer, using
* {@link java.lang.Integer#valueOf(String)}. Nulls are not allowed. Underlying
* type is
*/
public class Integer extends SimpleType implements Rangeable  {
  @Override
  public Object coerce(Object value) {
    checkForNull(value);

    if (value instanceof Number) {
      return ((Number)value).longValue();
    } else if (value instanceof String) {
      try {
        return new java.math.BigDecimal((java.lang.String) value).longValue();
      } catch (NumberFormatException ex) {
        throw new CoercionException(value, this);
      }
    } else {
      throw new CoercionException(value, this);
    }
  }

  @Override
  public Class<?> internalType() {
    return java.lang.Long.class;
  }

  @Override
  public int estimateSize(Object entry) {
    return java.lang.Integer.BYTES;
  }

  @Override
  public void toBytes(DataOutputStream os, Object toWrite) throws IOException {
    os.writeLong(((java.lang.Number) toWrite).longValue());
  }

  @Override
  public Object fromBytes(DataInputStream in) throws IOException {
    return in.readLong();
  }

  @Override
  public Object read(ByteBuffer buffer) {
    return buffer.getLong();
  }

  @Override
  public void write(ByteBuffer buffer, Object value) {
    buffer.putLong((Long) value);
  }

}
