/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.types;

import java.nio.ByteBuffer;

/**
 * Base for a {@link Type} that wraps {@link Types#GEOMETRY} to make it a more specific geometry
 * sub-type.
 */
public abstract class GeomType extends SimpleType implements WrappingType {

  public static class Line extends GeomType {
    @Override
    public Class<?> internalType() {
      return org.locationtech.jts.geom.LineString.class;
    }
  }

  public static class Point extends GeomType {
    @Override
    public Class<?> internalType() {
      return org.locationtech.jts.geom.Point.class;
    }
  }

  public static class Polygon extends GeomType {
    @Override
    public Class<?> internalType() {
      return org.locationtech.jts.geom.Polygon.class;
    }
  }

  /**
   * @return true if toCheck is assignable to geometry, false otherwise
   */
  public static boolean isAllowNull(Type toCheck) {
    return toCheck.findAllowNull(Geom.class).isPresent();
  }


  @Override
  public Type getUnderlyingType() {
    return Types.GEOMETRY;
  }

  @Override
  public Object coerce(Object value) throws CoercionException {
    Object coerced = getUnderlyingType().coerce(value);
    if (! internalType().isAssignableFrom(coerced.getClass())) {
      throw new CoercionException(value, this);
    }
    return coerced;
  }

  @Override
  public void write(ByteBuffer buffer, Object value) {
    Types.GEOMETRY.write(buffer, value);
  }

  @Override
  public Object read(ByteBuffer buffer) {
    return Types.GEOMETRY.read(buffer);
  }

  @Override
  public <T extends Type> boolean isA(Class<T> type) {
    return type.isAssignableFrom(getClass());
  }

  @Override
  public <T, U> U visit(TypeVisitor<T, U> tv, T data) {
    // despite the fact we are a simple type, we still wrap Geom - this gives us a sort of type hierarchy
    return WrappingType.super.visit(tv, data);
  }

}
