/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.google.common.base.Joiner;

import lombok.Getter;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.function.UntypedFunction;
import nz.org.riskscape.engine.rl.RealizableFunction;
import nz.org.riskscape.engine.util.Pair;

/**
 * Type information for a first-class function being stored in a tuple (as an {@link UntypedFunction})
 *
 * Does not support any type io or coercion operations
 */
public class FunctionType implements Type {

  /**
   * Wildcard style function type object, typically used with {@link RealizableFunction}s to say that some kind of
   * {@link FunctionType} will be accepted, or in type problems to report a broad mismatch, e.g. we wanted a function
   * type, but you gave us a list
   */
  public static final FunctionType WILD = new FunctionType(Collections.emptyList(), Types.ANYTHING);
  public static final Object VISITOR_META_RETURNS = "returns";

  public static FunctionType fromFunction(RiskscapeFunction function) {
    return new FunctionType(function.getArgumentTypes(), function.getReturnType());
  }

  private final Struct asStruct;
  @Getter
  private final List<Type> argumentTypes;
  @Getter
  private final Type returnType;

  /**
   * Construct a {@link FunctionType} with the given function signature of `argumentTypes` and `returnType`.  This
   * should mean that any covariant arguments can be safely applied to a function that has this type, and that returned
   * values will be covariant to the return type.
   */
  public FunctionType(List<Type> argumentTypes, Type returnType) {
    this.asStruct = Struct.of(SimpleType.DEFAULT_MEMBER_NAME, this);
    this.argumentTypes = argumentTypes;
    this.returnType = returnType;
  }

  @Override
  public Object coerce(Object value) throws CoercionException {
    throw new UnsupportedOperationException();
  }

  @Override
  public Class<?> internalType() {
    return UntypedFunction.class;
  }

  @Override
  public Struct asStruct() {
    return asStruct;
  }

  @Override
  public String toString() {
    return String.format("Function(ƒ(%s)=%s)", Joiner.on(", ").join(argumentTypes), returnType);
  }

  @Override
  public int estimateSize(Object entry) {
    throw new UnsupportedOperationException();
  }

  @Override
  public void toBytes(DataOutputStream os, Object toWrite) throws IOException {
    throw new UnsupportedOperationException();
  }

  @Override
  public Object fromBytes(DataInputStream in) throws IOException {
    throw new UnsupportedOperationException();
  }

  @Override
  public <T, U> U visit(TypeVisitor<T, U> tv, T data) {
    List<Pair<Type, ?>> children = new ArrayList<>(argumentTypes.size() + 1);

    for (Type childType : argumentTypes) {
      children.add(Pair.of(childType, TypeVisitor.NO_META));
    }

    children.add(Pair.of(returnType, VISITOR_META_RETURNS));

    return tv.compoundType(this, children, data);
  }

}
