/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Collections;
import java.util.List;

import lombok.Getter;

/**
 * A special case of a list type, which is always empty.  It is special, as it is co-variant with any other list, e.g.
 * an empty list can be assigned to any other list type without upsetting type safety.
 *
 * It wraps a list of type nothing, so that if you want to pretend it's a list, Type#find will do the right thing.  Note
 * that we return a list of type nothing so that if you attempt to use the member type, no equivalence or covariance is
 * going to work for you
 */
public class EmptyList implements Type, ContainingType, WrappingType {

  /**
   * The one and only instance of an EmptyList type
   */
  public static final EmptyList INSTANCE = new EmptyList();

  public static final Struct AS_STRUCT = Struct.of(SimpleType.DEFAULT_MEMBER_NAME, INSTANCE);

  @Getter
  private final Type underlyingType = RSList.create(Types.NOTHING);

  private EmptyList() {
  }

  @Override
  public Type getContainedType() {
    return Types.ANYTHING;
  }

  @Override
  public Object coerce(Object value) throws CoercionException {
    if (value instanceof List) {
      if (((List<?>) value).isEmpty()) {
        return value;
      }
    }
    throw new CoercionException(value, this, "can not coerce anything other than an empty list to an empty list");
  }

  @Override
  public Class<?> internalType() {
    return List.class;
  }

  @Override
  public Struct asStruct() {
    return AS_STRUCT;
  }

  @Override
  public int estimateSize(Object entry) {
    return 0;
  }

  @Override
  public void toBytes(DataOutputStream os, Object toWrite) throws IOException {
  }

  @Override
  public Object fromBytes(DataInputStream in) throws IOException {
    return Collections.emptyList();
  }

  @Override
  public String toString() {
    return "EmptyList()";
  }

  @Override
  public <T, U> U visit(TypeVisitor<T, U> tv, T data) {
    // XXX Not entirely sure what to do with empty list...  It could be an atom, a containing type, or a wrapping type.
    // I think from a purity point of view, it's an atom, but from the point of view of wanting to avoid a bunch of
    // special cases, maybe it's a containing type.  Might need to suck this and see a bit, and in practice it might not
    // matter too much, given it's something that typically only exists in constant expressions.
    return tv.atomicType(this, data);
  }
}
