/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;

/**
 * Permissive Decimal type that will convert any {@link java.lang.Number} in to
 * a BigDecimal via {@link Number#doubleValue()}. It will also attempt to
 * convert a {@link java.lang.String} to an integer, using
 * {@link java.math.BigDecimal#BigDecimal(String)}. Nulls are not allowed.
 */
public class Decimal extends SimpleType implements Rangeable {

  /**
   * A constant value of two.  Useful, and doesn't exist as a static member of BigDecimal already.
   */
  public static final BigDecimal TWO = new BigDecimal(2);

  /**
   * Too expensive to actually measure a {@link BigDecimal}, so just guess.
   */
  private static final int MAYBE_NUM_LONGS = 4;

  @Override
  public Object coerce(Object value) {
    checkForNull(value);

    if (value instanceof BigDecimal) {
      return value;
    } else if (value instanceof BigInteger) {
      return new BigDecimal((BigInteger) value);
    } else if (value instanceof Number) {
      return new BigDecimal(((Number) value).doubleValue());
    } else if (value instanceof java.lang.String) {
      try {
      return new BigDecimal(value.toString());
      } catch (NumberFormatException ex) {
        throw new CoercionException(value, this, "%s is not a number", value.toString());
      }
    } else {
      throw new CoercionException(value, this);
    }
  }

  @Override
  public Class<?> internalType() {
    return BigDecimal.class;
  }

  @Override
  public int estimateSize(Object entry) {
    // this seems good enough?
    return java.lang.Long.BYTES * MAYBE_NUM_LONGS;
  }

  @Override
  public void toBytes(DataOutputStream os, Object toWrite) throws IOException {
    BigDecimal value = (BigDecimal) toWrite;
    int scale = value.scale();
    byte[] unscaledValue = value.unscaledValue().toByteArray();

    os.writeInt(scale);
    os.writeInt(unscaledValue.length);
    os.write(unscaledValue);
  }

  @Override
  public Object fromBytes(DataInputStream in) throws IOException {
    int scale = in.readInt();
    int numBytes = in.readInt();

    byte[] integerBytes = new byte[numBytes];
    in.readFully(integerBytes);

    BigInteger integer = new BigInteger(integerBytes);
    return new BigDecimal(integer, scale);
  }

  @Override
  public Object read(ByteBuffer buffer) {
    int scale = buffer.getInt();
    int numBytes = buffer.getInt();

    byte[] integerBytes = new byte[numBytes];
    buffer.get(integerBytes);

    BigInteger integer = new BigInteger(integerBytes);
    return new BigDecimal(integer, scale);
  }

  @Override
  public void write(ByteBuffer buffer, Object toWrite) {
    BigDecimal value = (BigDecimal) toWrite;
    int scale = value.scale();
    byte[] unscaledValue = value.unscaledValue().toByteArray();

    buffer.putInt(scale);
    buffer.putInt(unscaledValue.length);
    buffer.put(unscaledValue);
  }
}
