/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Objects;

import lombok.Getter;
import nz.org.riskscape.engine.coverage.TypedCoverage;
import nz.org.riskscape.engine.typexp.AST;
import nz.org.riskscape.engine.typexp.AST.ComplexType;
import nz.org.riskscape.engine.typexp.ComplexTypeConstructor;
import nz.org.riskscape.engine.typexp.TypeArgumentException;
import nz.org.riskscape.engine.typexp.TypeBuilder;

/**
 * Represents a {@link TypedCoverage} using Riskscape's type system.  Gives information about what sort of
 * values the coverage will yield via {@link #getMemberType()}
 */
public class CoverageType implements Type, ContainingType {

  public static final ComplexTypeConstructor TYPE_CONSTRUCTOR = (TypeBuilder typeBuilder, ComplexType type) -> {
    List<AST> args = type.args();

    if (args.size() != 1) {
      throw new TypeArgumentException(type, "single argument expected");
    }

    Type subType = typeBuilder.expectType(args.get(0), ast
        -> new TypeArgumentException(type, "expected a type"));

    return new CoverageType(subType);
  };

  public static final TypeInformation TYPE_INFORMATION = new TypeInformation(
      "coverage",
      CoverageType.class,
      TypedCoverage.class,
      TYPE_CONSTRUCTOR
  );

  public static final CoverageType WILD = new CoverageType(Types.ANYTHING);

  /**
   * The {@link Type} of values returned from {@link TypedCoverage#evaluate(org.locationtech.jts.geom.Point)}
   */
  @Getter
  private Type memberType;

  public CoverageType(Type memberType) {
    this.memberType = memberType;
  }

  @Override
  public Object coerce(Object value) throws CoercionException {
    checkForNull(value);

    if (value instanceof TypedCoverage) {
      TypedCoverage cov = (TypedCoverage) value;
      if (cov.getType().equals(memberType)) {
        return value;
      } else {
        throw new CoercionException(
            value,
            this,
            "Give coverage's member type does not match - %s != %s",
            cov.getType(),
            memberType);
      }
    } else {
      throw new CoercionException(value, this, "Can not coerce object of type %s", value.getClass());
    }
  }

  @Override
  public Class<?> internalType() {
    return TypedCoverage.class;
  }

  @Override
  public Struct asStruct() {
    return Struct.of("value", this);
  }

  @Override
  public int estimateSize(Object entry) {
    return -1;
  }

  @Override
  public void toBytes(DataOutputStream os, Object toWrite) throws IOException {
    throw new UnsupportedOperationException();
  }

  @Override
  public Object fromBytes(DataInputStream in) throws IOException {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean equals(Object obj) {
    if (obj instanceof CoverageType) {
      return ((CoverageType) obj).memberType.equals(this.memberType);
    } else {
      return false;
    }
  }

  @Override
  public int hashCode() {
    return Objects.hash("coverage", memberType);
  }

  @Override
  public String toString() {
    return String.format("Coverage[%s]", this.memberType);
  }

  @Override
  public Type getContainedType() {
    return memberType;
  }

}
