/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.sort;

import com.google.common.collect.Lists;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.Constant;
import nz.org.riskscape.rl.ast.Expression;

/**
 * Defines an sort by, consisting of an {@link Expression} to obtain a comparable value and {@link Direction}.
 */
@RequiredArgsConstructor
@EqualsAndHashCode
public class SortBy {

  /**
   * Convenience version of {@link #concatenate(List)}
   */
  public static SortBy concatenate(SortBy... clauses) {
    return concatenate(Arrays.asList(clauses));
  }

  /**
   * Concatenate a list of {@link SortBy}s, return a new {@link SortBy} where the (n+1)th element is subordinate to the
   * nth.
   */
  public static SortBy concatenate(List<SortBy> clauses) {
    SortBy combined = null;
    for (SortBy sort: clauses) {
      if (combined == null) {
        combined = sort;
      } else {
        combined = combined.andThen(sort);
      }
    }
    return combined != null ? combined : SortBy.NONE;
  }

  public static final SortBy NONE = new SortBy(Constant.string(Boolean.toString(true)), Direction.ASC, null);

  public enum Direction {
    ASC,
    DESC
  }

  /**
   * Expression to extract value to sort by.
   */
  @Getter
  private final Expression expr;

  /**
   * The direction in which to sort.
   */
  @Getter
  private final Direction direction;

  /**
   * A subordinate, sort order
   */
  @Getter
  private final SortBy next;

  public SortBy(Expression expr, Direction dir) {
    this(expr, dir, NONE);
  }

  /**
   * Validate that tupleType would be able to be sorted by this SortBy.
   *
   * @param tupleType
   * @param expressionRealizer to realize expressions with
   * @return problems encountered, if any
   */
  public List<Problem> validate(Struct tupleType, ExpressionRealizer expressionRealizer) {
    if (this == NONE) {
      return Collections.emptyList();
    }

    List<Problem> problems = Lists.newArrayList(next.validate(tupleType, expressionRealizer));

    ResultOrProblems<RealizedExpression> type = expressionRealizer.realize(tupleType, expr);
    if (type.hasProblems(Problem.Severity.ERROR)) {
      problems.addAll(type.getProblems());
    } else {
      Type unwrapped = Nullable.strip(type.get().getResultType());
      if (!Comparable.class.isAssignableFrom(unwrapped.internalType())) {
        problems.add(Problem.error("Expression '%s' yields type '%s' which is not sortable",
            expr.toSource(), unwrapped));
      }
    }

    return problems;
  }

  /**
   * Builds a new SortBy clause where asNext is appended to the end of this {@link SortBy} chain.
   */
  public SortBy andThen(SortBy asNext) {
    if (this == NONE) {
      return asNext;
    } else {
      if (this.next == NONE) {
        return new SortBy(this.expr, this.direction, asNext);
      } else {
        return new SortBy(this.expr, this.direction, next.andThen(asNext));
      }
    }
  }

  @Override
  public String toString() {
    if (this == NONE) {
      return "SortBy(nothing)";
    } else {
      String nextString;
      if (next == NONE) {
        nextString = "";
      } else {
        nextString = next.toString();
      }

      StringBuilder builder = new StringBuilder(String.format("SortBy(%s %s),", this.expr, this.direction));
      builder.append(nextString);
      return builder.substring(0, builder.length() - 1);
    }
  }

}
