/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl.agg;

import java.util.Arrays;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.function.agg.AddAggregationFunction;
import nz.org.riskscape.engine.function.agg.AggregationFunctionAdapter;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

/**
 * A {@link RiskscapeFunction} sub-interface for a function that can be applied across many {@link Tuple}s as part of a
 * {@link RealizedAggregateExpression}.
 */
public interface AggregationFunction {

  /**
   * @return a new {@link RiskscapeFunction} that returns the given {@link AggregationFunction} when
   * {@link RiskscapeFunction#getAggregationFunction()} is called.  Useful for patching an existing function with an
   * aggregate version.
   */
  static RiskscapeFunction addAggregationTo(AggregationFunction setFunction, RiskscapeFunction wrapped) {
    return new AddAggregationFunction(wrapped, setFunction);
  }

  /**
   * @return a {@link RiskscapeFunction} that works on a list of the same elements supported by the given aggregation
   * function.
   */
  static RiskscapeFunction asFunction(final AggregationFunction aggregationFunction) {
    return new AggregationFunctionAdapter(aggregationFunction);
  }

  /**
   * Realize this function so that it can be called (via an Accumulator)
   * @param context the context it will be used under
   * @param fc the expression being realized
   * @return either a successfully realized function, or reasons why it could not be realized.
   */
  ResultOrProblems<RealizedAggregateExpression> realize(
      RealizationContext context,
      Type inputType,
      FunctionCall fc
  );

  /**
   * Returns the arguments this function expects, used for help and other introspection
   */
  default ArgumentList getArguments() {
    return ArgumentList.anonymous(Arrays.asList(RSList.LIST_ANYTHING));
  }

  /**
   * The return type this function produces, used for help and other introspection
   */
  default Type getReturnType() {
    return Nullable.ANYTHING;
  }
}
