/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl.agg;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.pipeline.Collector;

/**
 * Stores state during row-reduction as part of a {@link RealizedAggregateExpression}.  Has methods for applying state
 * from each row (typically a {@link Tuple}).
 *
 * Note that the {@link Accumulator} interface is not aware of any types or groups that were used or created when a
 * {@link RealizedAggregateExpression} was created - users of the API must only use accumulators with the expressions
 * that created them.
 */
public interface Accumulator {

  /**
   * Combine two accumulators in to one that records the state from both in a manner that gives the same result as if
   * each was reduced against a single {@link Accumulator}.  Exists to support parallelism in row reductions done as
   * part of the {@link Collector} interface.
   *
   * The returned object can be a new Accumulator or one of the existing ones.  It is assumed that the given
   * accumulators will not be used once this combine operation is completed.
   */
  Accumulator combine(Accumulator rhs);

  /**
   * Collect state in to this accumulator
   */
  void accumulate(Object input);

  /**
   * Convert accumulated state in to a single value.  Note that combine and accumulate should not be called again on an
   * {@link Accumulator} once process has been called.  This makes it safe for an accumulators state to be used in the
   * result or process without fear of mutability bugs
   * been called
   */
  Object process();

  /**
   * @return true if accumulate has never been used.  Expect `process` to return null (or throw an exception) if
   * isEmpty() is true.
   */
  boolean isEmpty();
}
