/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl;

import java.util.Collections;
import java.util.List;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.pipeline.Realized;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.rl.ast.Constant;
import nz.org.riskscape.rl.ast.Expression;

/**
 * Represents an Expression that has been realized against a specific RiskScape type.  Given an object that conforms to
 * that type, this expression can be safely evaluated against it.
 */
public interface RealizedExpression extends AutoCloseable {

  RealizedExpression TRUE = new RealizedExpression() {
    @Override
    public Type getResultType() {
      return Types.BOOLEAN;
    }

    @Override
    public Type getInputType() {
      return Types.ANYTHING;
    }

    @Override
    public Expression getExpression() {
      return Constant.TRUE;
    }

    @Override
    public Object evaluate(Object input) {
      return Boolean.TRUE;
    }

    @Override
    public List<RealizedExpression> getDependencies() {
      return Collections.emptyList();
    }

    @Override
    public String toString() {
      return "Constant(TRUE)";
    }

    @Override
    public void close() {
    }
  };

  /**
   * @return the type that this expression was realized for
   */
  Type getInputType();

  /**
   * @return the type that this expression yields
   */
  Type getResultType();

  /**
   * Evaluate this expression against the given object
   * @param input an object that must conform to {@link #getInputType()}
   * @return an object that must conform to {@link #getResultType()}
   */
  Object evaluate(Object input);

  /**
   * Convenience form of {@link #evaluate(java.lang.Object) } for expressions that you know accept a
   * {@link Struct} input type.
   *
   * This method will return `evaluate(Tuple.ofValues(getInputType().asStruct(), values)`
   */
  default Object evaluateValues(Object... values) {
    Tuple input = Tuple.ofValues(getInputType().asStruct(), values);

    return evaluate(input);
  }

  /**
   * @return the AST that this expression was realized from.
   */
  Expression getExpression();

  /**
   * @return any {@link RealizedExpression}s that are inputs in to this realized expression
   */
  List<RealizedExpression> getDependencies();

  /**
   * Destroy/release/abandon any resources associated with this expression.  Note that a realized expression should
   * propagate closing to any dependencies, as the default implementation does.
   *
   * See Javadoc {@link Realized} for more discussion on the life-cycle of executable objects like these.
   */
  @Override
  default void close() {
    for (RealizedExpression dependency : getDependencies()) {
      dependency.close();
    }
  }
}
