/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.resource;

import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;

import lombok.Data;

/**
 * A sort of transaction handle for saving resources to allow various housekeeping to be applied by various protocols
 * that allow storage.  Overkill for a filesystem API, but will be helpful/necessary for more sophisticated backends
 * and gives us a place to extend the API without requiring the {@link ResourceFactory#create(CreateRequest)} method to
 * change.
 */
@Data
public abstract class CreateHandle {

  @FunctionalInterface
  public interface Callback {
    void accept(OutputStream os) throws IOException;
  }

  /**
   * The original request made to save a resource
   */
  private final CreateRequest request;

  /**
   * Provide a callback that will store bytes associated with the resource.
   * @return URI a location where the successfully stored data can be accessed
   * @throws CreateException if something unforseen happened when trying to store bytes.  Typically this is going to be
   * out of the user's hands here, e.g. no space, connectivity error, etc
   */
  public URI store(Callback callback) throws CreateException {
    OutputStream os = getOutputStream();
    try {
      callback.accept(os);
    } catch (IOException e) {
      throw new CreateException(e);
    }
    return store();
  }

  /**
   * @return an {@link OutputStream} to use for writing data.
   */
  public abstract OutputStream getOutputStream() throws CreateException;

  /**
   * Store data written to the handle, returning the URI it is now available at.  If getOutputStream hasn't been
   * written too, strange things might happen when you call this (wowoooooo, spooky)
   */
  public abstract URI store() throws CreateException;

}
