/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.relation;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collector;

import com.google.common.collect.Streams;

import lombok.NonNull;
import nz.org.riskscape.engine.Tuple;

/**
 * Brings together the {@link AutoCloseable} and {@link Iterator} interface fo {@link Iterator}s that come from
 * {@link Relation}s, so that iterators can clean up any resources they are derived from.
 */
public interface TupleIterator extends Iterator<Tuple>, AutoCloseable {

  /**
   * An always empty {@link TupleIterator}, a bit like {@link Collections#emptyIterator()}
   */
  TupleIterator EMPTY = new TupleIterator() {

    @Override
    public Tuple next() {
      throw new NoSuchElementException();
    }

    @Override
    public boolean hasNext() {
      return false;
    }

    @Override
    public String toString() {
      return "Empty()";
    }
  };

  /**
   * Wraps a normal iterator with this interface, making the close a noop.
   * TODO - maybe add an optional callback for closing?
   */
  static TupleIterator wrapped(
      @NonNull Iterator<Tuple> iterator,
      @NonNull Optional<Runnable> onClose) {

    return new TupleIterator() {

      @Override
      public void close() {
        onClose.ifPresent(Runnable::run);
      }

      @Override
      public Tuple next() {
        Tuple value = iterator.next();
        return value;
      }

      @Override
      public void remove() {
        iterator.remove();
      }

      @Override
      public boolean hasNext() {
        return iterator.hasNext();
      }

      @Override
      public String toString() {
        return String.format("Wrapped(source=%s)", iterator);
      }
    };
  }

  /**
   * Wraps a possibly normal Iterator with this interface, applying the given mapping functions to values yielded from
   * the {@link Tuple}
   */
  static <T> TupleIterator wrappedAndMapped(
      @NonNull Iterator<T> iterator,
      @NonNull Function<T, Tuple> mapper) {

    return new TupleIterator() {

      @Override
      public Tuple next() {
        return mapper.apply(iterator.next());
      }

      @Override
      public void remove() {
        iterator.remove();
      }

      @Override
      public boolean hasNext() {
        return iterator.hasNext();
      }

      @Override
      public String toString() {
        return String.format("WrappedAndMapped(source=%s mapper=%s)", iterator, mapper);
      }
    };
  }

  @Override
  default void close() {
  }

  /**
   * Collect all values from this tuple iterator in to the given {@link Collector}, and ensures the iterator is closed.
   * @return the result of {@link Collection}
   */
  default <A, R> R collect(Collector<? super Tuple, A, R> collector) {
    return Streams.stream(this).collect(collector);
    // NB this used to close the iterator, but it seems the stream API is using forEachRemaining, which now
    // closes the TupleIterator.  There's a unit test that's asserting this is working.
  }

  /**
   * Pass each tuple to the given consumer, closing the iterator once complete (or if in exception is thrown)
   */
  @Override
  default void forEachRemaining(Consumer<? super Tuple> action) {
    try {
      Iterator.super.forEachRemaining(action);
    } finally {
      close();
    }
  }

  /**
   * @return a {@link TupleIterator} that yields a single, fixed result.
   */
  static TupleIterator singleton(Tuple tuple) {
    return new Singleton(tuple);
  }

}
