/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.relation;

import lombok.Getter;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.data.InputDataProblems;
import nz.org.riskscape.problem.Problem;

/**
 * Thrown when an invalid tuple (row) of input data is encountered, i.e. this indicates
 * there is a problem with the user's source data that needs addressing.
 */
@SuppressWarnings("serial")
public class InvalidTupleException extends RiskscapeException {

  @Getter
  private final Tuple tuple;

  /**
   * The specific reason why the tuple is invalid
   */
  @Getter
  private Problem reason;

  @Deprecated
  public InvalidTupleException(Tuple invalid, String message, Throwable cause) {
    this(invalid, Problem.error(message), cause);
  }

  @Deprecated
  public InvalidTupleException(Tuple invalid, String message) {
    this(invalid, Problem.error(message), null);
  }

  public InvalidTupleException(Tuple invalid, Problem problem) {
    this(invalid, problem, null);
  }

  private static Problem userFriendlyProblem(Problem problem) {
    // wrap the specific problem in a general-purpose, user-friendly tip for
    // what the user should do when an invalid tuple error is hit
    return InputDataProblems.get().invalidTupleError().withChildren(problem);
  }

  public InvalidTupleException(Tuple invalid, Problem problem, Throwable cause) {
    super(userFriendlyProblem(problem), cause);
    // store the specific reason the tuple is invalid, in case we need to refer back to it
    this.reason = problem;
    this.tuple = invalid;
  }

}
