/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.query;

import java.util.function.Supplier;

import lombok.NonNull;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Geom;
import nz.org.riskscape.engine.types.GeomType;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructMember;

public class TupleUtils {

  private static final int DEBUG_VALUE_MAX_LENGTH = 8;

  public enum FindOption {
    ONE_REQUIRED,
    ANY_REQUIRED,
    OPTIONAL
  }

  public static StructMember findGeometryMember(@NonNull Struct source, FindOption option) {
    return findGeometryMember(source, option, () -> new RiskscapeException(
        String.format("Failed to find geometry on %s according to option %s", source, option)));
  }
  /**
   * Look in the given struct for a member with type {@link Geom}.
   * @param source the type to search through
   * @param option flag whether at least on is required. If set to false, no exception is thrown if no
   * {@link Geom} member is found.
   * @throws IllegalArgumentException if either no or more than one {@link Geom} type was found in the given struct
   */
  public static StructMember findGeometryMember(
      @NonNull Struct source,
      @NonNull FindOption option,
      @NonNull Supplier<RuntimeException> failedHandler) {

    StructMember found = null;
    for (StructMember member : source.getMembers()) {
      if (GeomType.isAllowNull(member.getType())) {
        if (found == null) {
          found = member;
        } else if (option == FindOption.ONE_REQUIRED) {
          throw failedHandler.get();
        } else {
          return found;
        }
      }
    }

    if (found == null && option != FindOption.OPTIONAL) {
      throw failedHandler.get();
    } else {
      return found;
    }
  }

  /**
   * Convenience version of findGeometryMember where required flag is set to true
   */
  public static StructMember findRequiredGeometryMember(Struct struct) {
    return findGeometryMember(struct, FindOption.ONE_REQUIRED, ()
        -> new RiskscapeException("No single geometry member found on " + struct));
  }


  /**
   * Convenience version to check if the struct has a geometry member
   */
  public static boolean hasGeometryMember(Struct struct) {
    return findGeometryMember(struct, FindOption.OPTIONAL, ()
        -> new RiskscapeException("Internal error")) != null;
  }

  /**
   * @return a concise summary of the tuple's content. Values may be shortened, or
   *         in the case of geometries will be excluded completely.
   */
  public static String getContentSummary(Tuple tuple) {
    StringBuilder bldr = new StringBuilder("{");

    for (StructMember member : tuple.getStruct().getMembers()) {
      if (member.getType().findAllowNull(Geom.class).isPresent()) {
        // we skip geometry members
        continue;
      }
      if (bldr.length() != 1) {
        bldr.append(", ");
      }

      String value = member.getType().toString(tuple.fetch(member));
      if (value.length() > DEBUG_VALUE_MAX_LENGTH) {
        value = value.substring(0, DEBUG_VALUE_MAX_LENGTH - 1) + "…";
      }
      bldr.append(member.getKey()).append("=").append(value);
    }

    return bldr.append("}").toString();
  }
}
