/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.projection;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.pipeline.Realized;
import nz.org.riskscape.engine.query.TupleUtils;
import nz.org.riskscape.engine.relation.SpatialMetadata;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructMember;

/**
 * Realized {@link Projection} of one {@link Struct} to another.  Exists so that a Projection's projection function can
 * be associated with other metadata, such as the source and target {@link Struct} and a possible mapping of attributes
 * from one to the other.
 */
public interface Projector extends Function<Tuple, Tuple>, Realized {

  /**
   * Returns a 'noop' projector that does nothing other than pass the tuple along as-is
   * @param type the type of tuple that comes in and out
   */
  static Projector identity(Struct type) {
    return new Projector() {

      @Override
      public Tuple apply(Tuple t) {
        return t;
      }

      @Override
      public Struct getSourceType() {
        return type;
      }

      @Override
      public Struct getProducedType() {
        return type;
      }

      @Override
      public String toString() {
        return "IdentityProjector(%s)".formatted(type);
      }
    };
  }

  static String join(List<StructMember> members) {
    return members.stream().map(StructMember::getKey).collect(Collectors.joining("."));
  }

  static Map<List<StructMember>, List<StructMember>> identityMapping(Struct struct) {
    return struct.getMembers().stream().collect(
        Collectors.toMap(
          (m) -> Arrays.asList(m),
          (m) -> Arrays.asList(m)
        ));
  }

  /**
   * The exact {@link Struct} object that will be assigned to {@link Tuple}s that emerge from this {@link Projector}
   */
  @Override
  Struct getProducedType();

  /**
   * @deprecated use getProducedType
   */
  @Deprecated()
  default Struct getProjectedType() {
    return getProducedType();
  }

  /**
   * The exact {@link Struct} object that is expected to be seen by the
   * {@link #apply(java.lang.Object) }
   */
  Struct getSourceType();

  /**
   * @return a map that can be used to trace attributes from the target type (the key set) back to their source type
   * (value set).  Optional, as this is here to allow various optimisations to be applied to queries.  Not implementing
   * this method will prevent various optimisations for being possible.
   */
  default Map<List<StructMember>, List<StructMember>> getDirectMapping() {
    return Collections.emptyMap();
  }

  /**
   * An optional function used for transforming spatial metadata from source to target.  If left out, spatial metadata
   * is lost during projection.  A default, fairly sane implementation exists that copies the CRS and picks the single
   * geometry member that was found at the top level of the projectors projected type.
   */
  default Optional<Function<SpatialMetadata, SpatialMetadata>> getSpatialMetadataMapper() {
    return Optional.of((source) -> {
      if (source != null) {
        CoordinateReferenceSystem crs = source.getCrs();
        return Optional.ofNullable(TupleUtils.findGeometryMember(getProjectedType(),
            TupleUtils.FindOption.OPTIONAL))
            .map((sm) -> new SpatialMetadata(crs, sm))
            .orElse(null);
      } else {
        return null;
      }
    });
  }

  /**
   * @return a human-readable version of {@link #getDirectMapping()}, useful for debug statements, test assertions etc
   */
  default Map<String, String> getDirectMappingStrings() {
    return getDirectMapping().entrySet().stream().collect(
        Collectors.toMap(
            e -> join(e.getKey()),
            e -> join(e.getValue())
        ));
  }


}
