/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.plugin;

import java.nio.file.Path;
import java.util.Set;

/**
 * Contains meta data and standard methods for integrating a plugin's code base in to RiskScape
 */
public interface PluginDescriptor {

  /**
   * A unique id for the plugin - ideally globally unique, but must be at least install-unique
   */
  String getPluginId();

  /**
   * @return a class loader that can be used for loading i18n resource bundles
   */
  ClassLoader getI18nClassLoader();

  /**
   * @return if this plugin depends on other plugins
   */
  boolean hasPluginDependencies();

  /**
   * The plugin ids this plugin depends upon
   */
  Set<String> getPluginDependencies();

  /**
   * @return a newly constructed instance of this plugin
   */
  Plugin newPluginInstance();

  /**
   * Register a {@link PluginDescriptor} that is a dependency of this {@link PluginDescriptor} (according to
   * getPluginDependencies).
   *
   * This needs to happen before a plugin instance can be created via newPluginInstance as its likely this method will
   * set up class loading appropriately
   */
  void addDependency(PluginDescriptor dependencyDescriptor);

  /**
   * @return the java class name of the thing that implements Plugin
   */
  String getPluginClassName();

  /**
   * @return a version string for this plugin
   */
  String getVersion();

  /**
   * The location on the filesystem where this plugin's resources are located
   */
  Path getSourcePath();

}
