/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.plugin;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import nz.org.riskscape.engine.IdentifiedCollection;

/**
 * An {@link IdentifiedCollection} of {@link ExtensionPoint}s
 */
public class ExtensionPoints extends IdentifiedCollection.Base<ExtensionPoint<?>> {

  /**
   * @return a list of all the features registered to extension points with the given feature class
   */
  public <T extends PluginFeature> List<T> getFeaturesOfType(Class<T> featureClass) {
    return getAll().stream()
      .map(ep -> ep.checkFeatureClass(featureClass).map(ExtensionPoint::getFeatures).orElse(Collections.emptyList()))
      .flatMap(List::stream)
      .collect(Collectors.toList());
  }

  /**
   * Convenience method for adding an extension point by its feature class
   */
  public void addExtensionPoint(Class<? extends PluginFeature> featureClass) {
    add(new ExtensionPoint<>(featureClass));
  }

  /**
   * Add a feature to this set of extension points, returning false if none of the extension points supported this
   * feature.  Note that a feature can potentially implement multiple Feature SPIs, so can be added to more than one
   * ExtensionPoint
   */
  public boolean addFeature(PluginFeature feature) {
    boolean added = false;
    for (ExtensionPoint<?> extensionPoint : getAll()) {
      added = extensionPoint.addFeatureIf(feature) || added;
    }

    return added;
  }

}
