/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.plugin;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Identified;

/**
 * An ExtensionPoint is an official advertised place that allows a {@link Plugin} to advertise a {@link PluginFeature}
 * SPI and a place where implementing features should be registered by the plugin system during startup.
 *
 * ExtensionPoints are advertised by adding them to the list returned by {@link Plugin#getExtensionPoints()}
 *
 * TODO we *might* want to introduce some kind of bootstrapping thing? But that might in itself be a Feature. Going to
 * leave that out for now and see what we neeed.
 */
@RequiredArgsConstructor
public class ExtensionPoint<T extends PluginFeature> implements Identified {

  /**
   * The 'service provider interface' class for this extension. Any features that belong to this extension point are
   * instances of this class.
   *
   * Note that this doesn't actually have to be an interface, but it's standard nomenclature for this sort of thing.
   */
  @Getter
  private final Class<T> featureClass;

  /**
   * The list of implementing features that have been collected
   */
  private final List<T> collected = new ArrayList<>();

  @Override
  public String getId() {
    return featureClass.getCanonicalName();
  }

  /**
   * Add a feature to the list
   */
  public void addFeature(T feature) {
    collected.add(feature);
  }

  /**
   * Add a feature of unknown type to the list
   * @return true if it was added (and the right type)
   */
  public boolean addFeatureIf(PluginFeature feature) {
    if (featureClass.isInstance(feature)) {
      addFeature(featureClass.cast(feature));
      return true;
    } else {
      return false;
    }
  }

  /**
   * @return an immutable list of collected features for this extension point
   */
  public List<T> getFeatures() {
    return Collections.unmodifiableList(collected);
  }

  /**
   * @return this ExtensionPoint cast to the given feature class. Useful when working with a specific extension point
   *         and you need to get java's generic types to shut up and leave you alone.
   */
  @SuppressWarnings({ "rawtypes", "unchecked" })
  public <U extends PluginFeature> Optional<ExtensionPoint<U>> checkFeatureClass(Class<U> ofClass) {
    if (ofClass.equals(this.featureClass)) {
      ExtensionPoint thus = this;
      return Optional.of(thus);
    } else {
      return Optional.empty();
    }
  }

  @Override
  public String toString() {
    return String.format("ExtensionPoint(%s [%s])", featureClass.getSimpleName(), collected);
  }

}
