/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Range;
import com.google.common.collect.Sets;

import nz.org.riskscape.engine.Identified;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.ParameterSet;
import nz.org.riskscape.engine.i18n.RiskscapeMessage;
import nz.org.riskscape.engine.projection.FlatProjector;
import nz.org.riskscape.engine.projection.Projector;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.restriction.Restrictor;

/**
 * A processing {@link Step} that can be added to a {@link Pipeline}.   Note that steps themselves are not owned by a
 * pipeline and implementations should be immutable, relatively state-less and reusable.
 * A pipeline will bring steps together in a directed acyclic graph, and call the realize method in dependency order,
 * passing each realized step in to the next.
 *
 * Realization should return an object that implements exactly one of the {@link #FUNCTOR_INTERFACES} that is realized
 * using the given inputs.  This step should not perform any data processing, it should just validate the inputs against
 * its parameters and produce a functor that will accept data of the type it has been told it will receive (as per
 * the list of {@link RealizedStep}s it is given.
 *
 */
public interface Step extends Identified {

  /**
   * The supported set of interfaces a Step can produce during realization.
   */
  Set<Class<?>> FUNCTOR_INTERFACES = Sets.newHashSet(
      Relation.class,
      Restrictor.class,
      Projector.class,
      FlatProjector.class,
      Collector.class
  );

  /**
   * @return A human digestible description for this step.
   */
  RiskscapeMessage getDescription();

  /**
   * @deprecated use getParameterSet() instead
   */
  @Deprecated
  default List<Parameter> getDeclaredParameters() {
    return ImmutableList.copyOf(getParameterSet().getDeclared());
  };

  /**
   * The set of {@link Parameter}s this step accepts.  These should be used to populate a map for the
   * {@link #realize(RealizationInput)} method
   */
  ParameterSet getParameterSet();

  /**
   * Produce a new pipeline that is the result of realizing this step against the input.
   * @param stepInput the 'payload' for realization.
   */
  RealizedPipeline realize(RealizationInput stepInput);

  /**
   * @return a range that describes the number of inputs this step supports.
   * NB a bit unsure whether we really need this - at the moment it's defined by the functor... but it's possible
   * we might need a n-input join functor...
   */
  Range<Integer> getInputArity();

  default Optional<Parameter> getDeclaredParameter(String name) {
    return getDeclaredParameters().stream().filter(p -> p.getName().equals(name)).findFirst();
  }

  /**
   * @return a list of input names, in the order that the step expects. Will be empty for steps that do
   * not support named inputs.
   */
  default List<String> getInputNames() {
    return Collections.emptyList();
  }

  /**
   * @return the input name to attach un-named pipeline edges to, if named inputs are supported
   */
  default Optional<String> getDefaultInputName() {
    if (!getInputNames().isEmpty()) {
      return Optional.of(getInputNames().get(0));
    }
    return Optional.empty();
  }

  /**
   * @return true if this step has a named input that matches `namedInput`
   */
  default boolean hasNamedInput(String namedInput) {
    return getInputNames().contains(namedInput);
  }

  /**
   * @return a List of the names of parameters that this step declares
   */
  default List<String> getDeclaredParameterNames() {
    return getParameterSet().getDeclared().stream().map(Parameter::getName).collect(Collectors.toList());
  }

}
