/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import nz.org.riskscape.engine.Tuple;

/**
 * Accepts output tuples from a pipeline step, possibly for saving results, but also potentially for processing them
 * before having them emerge somewhere else in the pipeline.
 *
 * Some of this interface is coded for potential asynchronous I/O, but for now we're not respecting it (for the most
 * part) and neither are we using it.  But the idea is that in the future that instead of blocking, a Sink can refuse
 * to accept more {@link Tuple}s if some underlying buffer is full.
 *
 * Note that a Sink doesn't extend {@link AutoCloseable} - finish has different semantics to close - finish means we've
 * stopped writing to it, but it doesn't (necessarily) mean we're ready to do away with the sunk results.  So for now
 * at least they are separate.
 */
public interface Sink {

  /**
   * A sink that throws tuples away
   */
  Sink DEVNULL = new Sink() {

    @Override
    public boolean accept(Tuple tuple) {
      return true;
    }

    @Override
    public void finish() {
    }
  };

  /**
   * Pass a tuple to the sink to consume.
   * @return false if the sink is full and can not accept the tuple (right now)
   */
  boolean accept(Tuple tuple);

  /**
   * Tells the sink that there are no more tuples left to accept.
   */
  void finish();

  /**
   * If true, this sink will never return false from {@link #accept(Tuple)}
   */
  default boolean isUnbounded() {
    return true;
  }

  /**
   * If this sink is a bounded sink, then this method should return a best guess at whether accept will return true.
   * Calling code should be prepared for `canAccept` to return true but then the call to `accept` will return false.
   *
   */
  default boolean canAccept() {
    return true;
  }

}
