/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import nz.org.riskscape.ReflectionUtils;
import nz.org.riskscape.ReflectionUtils.SearchOptions;
import nz.org.riskscape.engine.types.Struct;

/**
 * Interface for other interfaces that can be returned from a {@link Step#realize(RealizationInput)}.
 *
 * A {@link Realized} object should be relatively cheap to construct and should avoid performing any up-front
 * computation based on its parameters.  Expensive objects should be lazily created using the ExpensiveResource wrapper
 * or some other mechanism.  Similarly, these objects can be cleaned up via the close method.
 *
 * For more discussion on the life-cycle of executable pipeline objects, see GL605
 */
public interface Realized extends AutoCloseable {

  /**
   * Utility for pulling which {@link Realized} sub-interface is being implemented by a class that implements realized
   * class
   */
  static Class<? extends Realized> getRealizedInterface(Class<? extends Realized> clazz) {
    return ReflectionUtils.findImplementingClass(Realized.class, clazz, SearchOptions.SEARCH_INTERFACES);
  }

  /**
   * Clean up any resources allocated to this {@link Realized} thing.  This might involve removing temporary files,
   * closing or flushing a network connection to some external service, or terminating some child processes.  Note that
   * it's expected that this object won't be used again after being closed.
   *
   * In some instances, implementations may want to remove references to large (in terms of memory) objects so that they
   * can be garbage collected even if references to the pipeline continue to keep them 'live' in the JVM's object graph
   */
  @Override
  default void close() {
  }

  /**
   * The type of tuple that come out of this realized thing, or {@link Struct#EMPTY_STRUCT} if nothing
   */
  Struct getProducedType();
}
