/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import java.util.List;
import java.util.Map;

import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.pipeline.ast.StepDefinition;

/**
 * An aggregate of the current state of pipeline realization to simplify the {@link Step#realize(RealizationInput)}
 * method.
 */
public interface RealizationInput {

  /**
   * @return the context in which any realized pipeline will eventually execute
   */
  default ExecutionContext getExecutionContext() {
    return getRealizedPipeline().getContext();
  }

  /**
   * @return The part of the AST that is being realized.
   */
  StepDefinition getStepDefinition();

  /**
   * A unique name (within the pipeline) assigned to the step being realized.  It is assumed that realizing will
   * introduce at least a step with this name in to the pipeline.
   */
  String getName();

  /**
   * @return the list of realized steps that this step depends on, i.e. they are chained to this step.
   */
  List<RealizedStep> getDependencies();

  /**
   * @return parameters already bound to this step via the ast
   */
  Map<String, List<?>> getParameters();

  /**
   * Returns the state of the pipeline "going in" to realization.  The {@link Step} implementation doing the realizing
   * should be adding new {@link RealizedStep}s to this pipeline.
   */
  RealizedPipeline getRealizedPipeline();

  /**
   * @return the context in which binding occurs, linked to the execution context
   */
  default BindingContext getBindingContext() {
    return getExecutionContext().getBindingContext();
  }

  /**
   * @return the context in which realization occurs, linked to the execution context
   */
  default RealizationContext getRealizationContext() {
    return getBindingContext().getRealizationContext();
  }

  /**
   * @return an incomplete RealizedStep that is suitable to be modified and returned by a 'simple' Step implementation.
   */
  default RealizedStep newPrototypeStep() {
    return RealizedStep.fromInput(this);
  }
}
