/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import java.util.List;

import nz.org.riskscape.dsl.SourceLocation;
import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.pipeline.ast.StepDefinition;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

/**
 * Factory for producing problems that involve pipelines or pipeline steps.
 * See also {@link ArgsProblems} for dealing with step argument problems.
 */
public interface PipelineProblems extends ProblemFactory {

  static PipelineProblems get() {
    return Problems.get(PipelineProblems.class);
  }
  /**
   * Assigned to the result of {@link RealizedStep#getFailureProblem() as the
   * parent problem}. Produces message: "Could not realize <step>"
   */
  Problem cannotRealize(StepDefinition step);

  Problem cannotRealize(Class<?> thing);

  /**
   * Step could not be realized because of failed dependencies
   */
  Problem failedDependencies(String dependentStepName);

  /**
   * Attempt to chain to a named input that doesn't exist on the target step.
   * Produces message:
   *   <step> has no input named '<inputName>'. Known input names are <supportedInputs>
   */
  Problem namedInputUnknown(Step step, String inputName, List<String> supportedInputs);

  /**
   * Problem associated with an {@link ExecutionException}
   */
  Problem executionFailed();

  /**
   * Attached to an OutOfStorageException
   */
  Problem outOfStorage(String storageLocation, long amountFree);

  /**
   * There was a cycle in your pipeline
   * @param offendingLink the link in the pipeline that when added caused the cycle, e.g. foo() -> bar()
   * @param linkLocation the line the link was added on, uses the location of the `->` token
   * @param offendingDeclaration The `->` and step declaration on the rhs of the link which if removed will remove the
   * cycle
   * @param firstOccurence the place where we first saw the offendingDeclaration
   */
  Problem cycleDetected(
    String offendingLink,
    SourceLocation linkLocation,
    String offendingDeclaration,
    SourceLocation firstOccurence
  );

  /**
   * The pipeline contains two steps with the same name.
   * @param stepName            the name that is being reused
   * @param originalLocation    the line number that stepName was first defined
   * @param redefinedAt         the line number that is attempting to redefine stepName
   */
  Problem stepRedefinition(
      String stepName,
      SourceLocation originalLocation,
      SourceLocation redefinedAt
  );

  /**
   * Attempt to chain the target step from the source step which produces no output.
   */
  Problem chainingFromStepWithNoOutput(String step, List<String> dependents);
}
