/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import com.google.common.collect.Range;

import nz.org.riskscape.engine.bind.ParameterSet;
import nz.org.riskscape.engine.i18n.RiskscapeMessage;
import nz.org.riskscape.engine.relation.EmptyRelation;
import nz.org.riskscape.engine.restriction.Restrictor;
import nz.org.riskscape.engine.types.Struct;

/**
 * A {@link Pipeline} {@link Step} that does nothing.  It is mostly here for completeness and is useful in cases where
 * a null might want to be returned or where something needs to be present that isn't an actual useful thing.  If
 * realized, this step produces an empty relation that yields an empty struct.
 *
 * TODO We may want to tweak it slightly so that if given input it yields
 * a {@link Restrictor} that always says no.
 *
 */
public class NullStep implements Step {

  public static final NullStep INSTANCE = new NullStep();

  @Override
  public ParameterSet getParameterSet() {
    return ParameterSet.EMPTY;
  }

  @Override
  public Range<Integer> getInputArity() {
    return Range.all();
  }

  @Override
  public RealizedPipeline realize(RealizationInput input) {
    return input.getRealizedPipeline().add(
        input.newPrototypeStep().realizedBy(this).withResult(new EmptyRelation(Struct.EMPTY_STRUCT))
    );
  }

  @Override
  public String getId() {
    return "null";
  }

  @Override
  public RiskscapeMessage getDescription() {
    return RiskscapeMessage.ofDefault("N/A");
  }

  @Override
  public String toString() {
    return getId();
  }

}
