/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import java.nio.file.Path;

import com.codahale.metrics.MetricRegistry;

import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration;

/**
 * Various bits associated with the realization and execution of a pipeline
 */
public interface ExecutionContext extends AutoCloseable {

  Engine getEngine();
  Project getProject();
  ExpressionRealizer getExpressionRealizer();
  BindingContext getBindingContext();
  MetricRegistry getMetricRegistry();
  PipelineExecutor getPipelineExecutor();
  ExecutionContext createNestedContext(String name);

  /**
   * @return a directory that can be used for temporary files for the duration of this execution context.
   */
  Path getTempDirectory();

  /**
   * Closes the execution context and removes any temporary resources created within it.
   */
  @Override
  void close();

  /**
   * Returns a {@link RealizationContext} bound to the {@link ExecutionContext} for use when realizing expressions for
   * use within this {@link ExecutionContext}.
   *
   * At the moment, this is just a handy accessor, but in the future, any resources used or created during realization
   * might be managed or dependent on the execution context, so in time we want to make sure that any expressions
   * realized for use within a pipeline use this context, instead of creating their own 'one off' context object.
   */
  default RealizationContext getRealizationContext() {
    return getBindingContext().getRealizationContext();
  }

  /**
   * @return a PipelineRealizer that can be used to realize a pipeline.
   */
  PipelineRealizer getPipelineRealizer();

  /**
   * Convenience method for realizing a pipeline using this {@link ExecutionContext}.
   * See {@link PipelineRealizer} for more details
   */
  default RealizedPipeline realize(PipelineDeclaration ast) {
    return getPipelineRealizer().realize(this, ast);
  }

}
