/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import java.util.concurrent.TimeUnit;

import com.codahale.metrics.Gauge;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.relation.TupleIterator;

/**
 * Implements a simple single-threaded collection of results from a Collector
 */
@RequiredArgsConstructor
public class DefaultCollection<T> implements Gauge<String> {

  /**
   * Delays accumulation and processing until next or hasNext is called
   */
  private class LazyIterator implements TupleIterator {

    private final TupleIterator collectFrom;
    private TupleIterator built;

    LazyIterator(TupleIterator collectFrom) {
      this.collectFrom = collectFrom;
    }
    @Override
    public boolean hasNext() {
      if (built == null) {
        built = DefaultCollection.this.collect(collectFrom);
      }
      return built.hasNext();
    }

    @Override
    public Tuple next() {
      if (built == null) {
        built = DefaultCollection.this.collect(collectFrom);
      }
      return built.next();
    }
  }

  public TupleIterator lazilyCollected(TupleIterator collect) {
    return new LazyIterator(collect);
  }

  private final Collector<T> collector;
  private String state = "waiting";
  private Timer accumulationTimer = new Timer();

  public TupleIterator collect(TupleIterator toCollect) {
    T accumulator = collector.newAccumulator();

    state = "started";

    while (toCollect.hasNext()) {
      Tuple collectThis = toCollect.next();
      long nanoStart = System.nanoTime();
      collector.accumulate(accumulator, collectThis);
      accumulationTimer.update(System.nanoTime() - nanoStart, TimeUnit.NANOSECONDS);
    }
    state = "finished";
    return collector.process(accumulator);
  }

  public void addMetrics(MetricRegistry registry, String prefix) {
    registry.register(prefix + "accumulation", accumulationTimer);
    registry.register(prefix + "state", this);
  }

  @Override
  public String getValue() {
    return state;
  }
}
