/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import java.util.Collections;
import java.util.Optional;
import java.util.Set;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.relation.TupleIterator;
import nz.org.riskscape.engine.types.Struct;

/**
 * Realized interface for a functor that can arbitrarily transform one set of tuples in to another.
 */
public interface Collector<T extends Object> extends Realized {

  /**
   * A set of characteristics a collector can have which allow the {@link PipelineExecutor} to better optimise and
   * organize execution.
   *
   * Currently only PARALLELIZABLE exists, but others might be added, such as whether accumulators share state (so there
   * will be some level of locking).
   */
  enum Characteristic {
    /**
     * The collector can be parallelized by creating multiple accumulators and combining them before processing.
     */
    PARALLELIZABLE,
  }

  /**
   * The type of tuple this collector accepts.
   */
  Struct getSourceType();

  /**
   * The type of tuple this collector produces
   */
  @Override
  Struct getProducedType();

  /**
   * @deprecated use {@link #getProducedType()}
   */
  @Deprecated
  default Struct getTargetType() {
    return getProducedType();
  }

  /**
   * Create a new accumulator that holds collection/aggregation state for this collector to use.  These don't need
   * to be thread safe - any code making use of this API should assume they'll be accessed by only a single thread.
   * NB do we want to support a shared accumulator API?
   */
  T newAccumulator();

  /**
   * Give a tuple to the accumulator for accumulation.  This should add any required state to the accumulator for this
   * tuple, for example add a number to a running total, append an element to a list, etc.
   */
  void accumulate(T accumulator, Tuple tuple);

  /**
   * Combine two accumulators, returning an accumulator that contains the accumulated state of both.  This needs to be
   * done in such a way that it produces the same result as if only one accumulator is used.  Implementations are free
   * to modify and return one of the given accumulators if it makes sense.
   */
  T combine(T lhs, T rhs);

  /**
   * Convert an accumulator in to a set of results.  This can be done lazily, e.g. rows can be pulled from the
   * accumulator on demand.
   */
  TupleIterator process(T accumulator);

  Class<T> getAccumulatorClass();

  /**
   * @return a set of any {@link Characteristic}s this {@link Collector} has
   */
  default Set<Characteristic> getCharacteristics() {
    return Collections.emptySet();
  }

  /**
   * @return true if this Collector has the {@link Characteristic#PARALLELIZABLE}.
   */
  default boolean isParallelizable() {
    return getCharacteristics().contains(Characteristic.PARALLELIZABLE);
  }

  /**
   * @return the number of accumulated entries, i.e. the total number of tuples
   *         that {@link #process(Object)} is expected to emit.
   */
  default Optional<Long> size(T accumulator) {
    return Optional.empty();
  }
}
