/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.output;

import java.net.URI;

import com.codahale.metrics.MetricRegistry;

import nz.org.riskscape.engine.Identified;
import nz.org.riskscape.engine.pipeline.RealizedPipeline;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Defines an output store that is able to store the results of pipeline execution, aka a pipeline job.
 */
public interface PipelineOutputStore extends Identified {

  int PRIORITY_NA = 0;
  int PRIORITY_DEFAULT = 1;
  int PRIORITY_HIGH = 1000;

  /**
   * @return a new {@link PipelineOutputOptions} object suitable for use with this store.  May contain defaults or
   * extra settings specific to this store, or just be a default {@link PipelineOutputOptions} object.
   */
  default PipelineOutputOptions newOutputOptions(String reference) {
    return new PipelineOutputOptions(reference);
  }

  /**
   * Returns a score indicating if this store is able to create a {@link PipelineOutputContainer} for
   * the given URI.
   *
   * Note that this method need only check that the URI can be handled by the store, i.e. it has a recognized scheme.
   * By the same token , returning a value greater than zero does not mean
   * {@link #create(URI, RealizedPipeline, MetricRegistry, PipelineOutputOptions) } is guaranteed to return a successful
   * result for the same URI.
   *
   * @param outputLocation a non-null user specified output location
   * @return a value greater than zero if a this store can handle the given URI.
   */
  int isApplicable(URI outputLocation);

  /**
   * Creates a {@link PipelineOutputContainer} to store outputs from executing the given pipeline.
   *
   * @param outputLocation    URI indicating where outputs should be stored
   * @param pipeline          the pipeline whose outputs are to be stored
   * @param options           user-defined, pipeline-global options to customize how output is stored
   * @return a {@link PipelineOutputContainer} or problems describing why it couldn't be created
   */
  ResultOrProblems<PipelineOutputContainer> create(
      URI outputLocation,
      RealizedPipeline pipeline,
      PipelineOutputOptions options
  );

}
