/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.output;

import java.net.URI;
import java.nio.file.Path;
import java.util.Map;

import com.codahale.metrics.MetricRegistry;

import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.pipeline.Manifest;
import nz.org.riskscape.engine.pipeline.RealizedStep;
import nz.org.riskscape.engine.pipeline.Sink;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * A container for storing outputs from a pipeline run.
 *
 * In addition to storing outputs created from {@link #createSinkForStep(RealizedStep) } the output container
 * should also store the:
 * - {@link PipelineDeclaration} (if available)
 * - {@link Manifest}
 * - progress metrics {@link MetricRegistry}
 */
public interface PipelineOutputContainer extends AutoCloseable {

  /**
   * Create a {@link Sink} for the given terminalStep.  This sink can then be used to 'save' tuples that form a model
   * output.  The format they are written in depends on what the {@link PipelineOutputStore} supports.
   * @param parameters to tailor the created sink
   * @return sink or problems preventing one from being created
   */
  ResultOrProblems<Sink> createSinkForStep(SinkParameters parameters);

  /**
   * Registers a local file to be stored as part of the pipeline's outputs.  The given file will potentially be moved
   * or copied in to a separate location (alongside the other outputs) when the container is closed.
   */
  default void registerLocalFile(Path localFile) {
    throw new RiskscapeException(GeneralProblems.get().operationNotSupported("registerOutput", getClass()));
  }

  /**
   * Finish writing to the output container. At this point all {@link Sink}s should have been closed.
   *
   * If the store supports writing the progress metrics now is the time to write them.
   */
  @Override
  void close();

  /**
   * @return a map of output names to storage locations.
   * TODO we might want to replace URI with something a bit more informative, such as number of rows, size, checksum etc
   */
  Map<String, URI> getPipelineOutputs();

  /**
   * @return the store that created this container.
   */
  PipelineOutputStore getStore();

  /**
   * Returns a URI that describes where the outputs are stored.  Note this relates to the pipeline output URIs in that
   * this URI is typically a 'container' URI, but how these relate will typically depend on the format or type of
   * storage
   *
   * Note that at this stage we're not making any guarantees that this will be generally useful beyond being
   * informational.  Ideally, other components could reason about these URIs based on their structure and do something
   * useful with them, e.g. if it looks like a local directory, you could open it.  If it's a database URL, you could
   * attempt to connect to it.
   *
   * Note also that this URI might be different to the one the user gave,
   * e.g. we might add a bunch of directories to the beginning of a file, or resolve/follow a URI to some other
   * location, so this should be used in preference to the one the user gave when presenting info back to the user
   */
  URI getStoredAt();
}
