/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.output;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import nz.org.riskscape.engine.Identified;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * A Format represents a file format (or other) used for data coming in and out of RiskScape.  This could be something
 * simple like
 * a CSV file or bitmap data coming from a WCS server.
 *
 * Format is set to become the central place where data formats are registered and as a jumping off point for other
 * APIs that allows reading and writing of those formats.  For now we are using this for creating RiskscapeWriter only,
 * but we will want to evolve this and make Format a part of the BookmarkResolver API, i.e. add a getFormat() method
 * to BookmarkResolver.  Also note that format should not be bound to row-based formats and that we want to make sure
 * we allow things like coverages or bitmaps to come on board, too
 */
public interface Format extends Identified {

  /**
   * As yet empty format options that apply to all formats.
   */
  class FormatOptions {}

  /**
   * Format characteristics are here to guide the various bits of code that use them to be able to use them generically
   * without prior knowledge of them specifically.  For example, we might want to only allow the use of STREAMABLE
   * formats with a specific output backend.  Or restrict the use of MULTIPLE_FILES formats to the file protocol only.
   */
  enum Characteristics {

    /**
     * This format can read/write tuples from a stream of bytes without needing all the contents at once
     */
    STREAMABLE,

    /**
     * This format ends up storing multiple files, rather than a single one.  These are a mess and we might want to
     * impose some conventions on these, e.g. they end up in a zip file, or a folder
     */
    MULTIPLE_FILES,

    /**
     * This format doesn't read/write bytes to some sort of file (or files), e.g. it uses some sort of protocol (e.g. an
     * SQL connection) to read and write tuples.  Formats that end up writing bytes to a file shouldn't use this.
     * TODO maybe this doesn't exist for Format?  Maybe that's something else?
     */
    PROTOCOL_BASED,

    /**
     * This format doesn't require geometry to be present and might not record spatial data efficiently (or accurately)
     */
    NON_SPATIAL,
  }

  /**
   * @return the file extension that this format is normally saved as
   */
  String getExtension();

  /**
   * @return the media type for this format
   */
  String getMediaType();

  /**
   * @return the set of characteristics associated with this data format.
   */
  Set<Characteristics> getCharacteristics();

  /**
   * @return the {@link Class} defining writer options for this format. Objects of this class may be
   * used as the format
   */
  default Class<? extends FormatOptions> getWriterOptionsClass() {
    return FormatOptions.class;
  }

  /**
   * Builds a FormatOptions object from the given parameter map
   */
  ResultOrProblems<? extends FormatOptions> buildOptions(
          Map<String, List<?>> paramMap, BindingContext context, Struct input);

  /**
   * Returns options that are required for this output format (if any)
   */
  default List<String> getRequiredOptions(BindingContext context) {
    return Collections.emptyList();
  }

  /**
   * Get a {@link WriterConstructor} to create an output writer with.
   * @return the writer constructor or empty if writing tuples to this format is not supported
   */
  Optional<WriterConstructor> getWriterConstructor();

}
