/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.model;

import java.util.List;
import java.util.Locale;
import java.util.Set;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.bind.BoundParameters;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.ParameterProperty;
import nz.org.riskscape.engine.bind.ParameterTemplate;
import nz.org.riskscape.engine.bind.ParameterTemplate.Choice;

/**
 * Represents a {@link Parameter} in a {@link Model} that can be customized by
 * the user, via a {@link ParameterTemplate}
 */
@RequiredArgsConstructor
public class ModelParameter {

  @Getter
  private final Parameter parameter;
  @Getter
  private final ParameterTemplate template;

  /**
   * @return a detailed user-friendly description of the parameter, i.e. help text
   */
  public String getDescription(Locale locale) {
    return template.getDescription().orElse("");
  }

  /**
   * @return a user-friendly label for this parameter, like what would be displayed in the UI
   */
  public String getLabel() {
    return template.getLabel().orElse(toUserFriendlyString(parameter.getName()));
  }

  protected String toUserFriendlyString(String input) {
    // capitalize first letter and replace hyphens/underscores with spaces
    return input.substring(0, 1).toUpperCase() + input.substring(1).replace("-", " ").replace("_", " ");
  }

  /**
   * @return any traits that this parameter has, e.g. hidden, read-only, numeric, etc
   */
  public Set<ParameterProperty> getProperties() {
    return template.getProperties();
  }

  /**
   * @return a set of pre-defined values that this parameter can accept
   */
  public List<Choice> getChoices(Locale locale) {
    return template.getChoices();
  }

  public Object getValue(BoundParameters bound) {
    return bound.getValue(getName());
  }

  public String getName() {
    return parameter.getName();
  }
}
