/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.lookup;

import nz.org.riskscape.engine.data.SelfDescribingScalarData;
import nz.org.riskscape.engine.types.LookupTableType;
import nz.org.riskscape.engine.types.Type;

/**
 * A LookupTable is a data source that allows some value to be looked up based on a key.  It is assumed that the values
 * are looked up with some kind of indexing.
 */
public interface LookupTable extends SelfDescribingScalarData {

  /**
   * @return type of values returned by lookups.  Note that this shouldn't declare nullable unless keyed values in the
   * table can be null
   */
  Type getValueType();

  /**
   * @return type of keys used to lookup values
   */
  Type getKeyType();

  /**
   * @return A {@link LookupTableType} that represents this {@link LookupTable}.
   */
  default LookupTableType getLookupTableType() {
    return new LookupTableType(getKeyType(), getValueType());
  }

  /**
   * Lookup a value in this table, or null if there is no value for the given key (or null if the key mapped to null)
   */
  Object lookup(Object key);

  @Override
  default Type getScalarDataType() {
    return new LookupTableType(getKeyType(), getValueType());
  }
}
