/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.join;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.pipeline.Realized;
import nz.org.riskscape.engine.relation.Singleton;
import nz.org.riskscape.engine.relation.TupleIterator;
import nz.org.riskscape.engine.types.Struct;

/**
 * A variation on Joiner that works by joining individual values from the lhs (left-hand side) to an indexed
 * set of rhs (right-hand side) values.  This API expects that the work of building the index be done for it in advance
 * of being passed to {@link #join(Object, Tuple)}
 */
public interface LopsidedJoiner<T> extends Realized {

  int LHS_STEP_INDEX = 0;
  int RHS_STEP_INDEX = 1;

  /**
   * Expected type of the lhs of this join
   */
  Struct getLhs();

  /**
   * Expected type of the rhs of this join
   */
  Struct getRhs();

  /**
   * The type of tuple that {@link #join(Object, Tuple)} produces.
   */
  @Override
  Struct getProducedType();

  /**
   * @deprecated use {@link #getProducedType()}
   */
  @Deprecated
  default Struct getJoinedType() {
    return getProducedType();
  }

  /**
   * @return a new, empty index, for use with {@link #addToRhsIndex(Tuple, Object)}
   */
  T createRhsIndex();

  /**
   * Accumulate a rhs value in to the index
   */
  void addToRhsIndex(Tuple rhs, T index);

  /**
   * Join a single lhs value to the indexed set of rhs values.  Implementations should use {@link TupleIterator#EMPTY}
   * and {@link Singleton} where possible to take advantage of various optimisations in pipeline execution.
   */
  TupleIterator join(T index, Tuple lhs);
}
