/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import java.util.Locale;

import lombok.Data;

/**
 * Represents the current context of the translation, i.e. what locale/language is being used, and
 * the translated message strings we're interested in.
 *
 * This allows objects to have more finer-grain control over their translation. Instead of having to
 * generate a {@link MessageKey} that someone else then uses to lookup the {@link MessageSource}
 * based on the current {@link Locale}, the object can use the TranslationContext to return the
 * translated string itself.
 */
@Data
public class TranslationContext {

  /**
   * The locale being used for the translation
   */
  private final Locale locale;

  /**
   * The MessageSource being used for translation (i.e. contains the message text indexed by code for
   * a given resource bundle).
   */
  private final MessageSource messageSource;

  /**
   * @return a new {@link MessageFactory} instance that returns translated messages against the given locale
   * and message source
   */
  public <T extends MessageFactory> T newFactory(Class<T> factoryInterface) {
    return messageSource.getMessageFactory(factoryInterface, locale);
  }

  /**
   * Allows this object to act a little like a MessageSource, which can help with
   * statements().that().look().like().this();
   */
  public String getMessage(String code, Object... args) {
    return messageSource.getMessage(code, args, locale);
  }

  /**
   * Same theory as {@link #getMessage(String code, Object... args)} but with a default message
   */
  public String getMessage(String code, Object[] args, String defaultMessage) {
    return messageSource.getMessage(code, args, defaultMessage, locale);
  }
}
