/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.i18n;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NonNull;

@EqualsAndHashCode
public class RiskscapeMessage implements MessageKey {

  /**
   * Build a {@link RiskscapeMessage} with desired code and messageArguments.
   * @param code
   * @param arguments
   * @return RiskscapeMessage  with desired code and messageArguments
   */
  public static RiskscapeMessage of(@NonNull String code, Object... arguments) {
    return new RiskscapeMessage(code, arguments, null);
  }

  /**
   * Build a {@link RiskscapeMessage} with desired code and messageArguments.
   * @param code
   * @param defaultMessage
   * @param arguments
   * @return RiskscapeMessage with default code and messageArguments
   */
  public static RiskscapeMessage withDefault(@NonNull String code, @NonNull String defaultMessage,
      Object... arguments) {
    return new RiskscapeMessage(code, arguments, defaultMessage);
  }

  /**
   * Build a {@link RiskscapeMessage} that only contains the default message.
   *
   * Useful when API's require a {@link RiskscapeMessage} to be used but there are no reasonable codes
   * to use. Such as when the message actually comes from user provided sources.
   */
  public static RiskscapeMessage ofDefault(@NonNull String defaultMessage, Object... args) {
    return new RiskscapeMessage("", args, defaultMessage);
  }

  @Getter
  private final String code;
  @Getter
  private final Object[] messageArguments;
  @Getter
  private final String defaultMessage;

  private RiskscapeMessage(String code, Object[] messageArguments, String defaultMessage) {
    this.code = code;
    this.messageArguments = messageArguments != null ? messageArguments : new Object[]{};
    this.defaultMessage = defaultMessage;
  }

}
