/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Consumer;

import lombok.Data;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.OsUtils;
import nz.org.riskscape.problem.Problem;

/**
 * Returned from {@link Messages#renderProblem(Problem, java.util.Locale)} to translate a problem while maintaining its
 * tree structure.  This should allow UIs to present it appropriately without having to translate it all individually.
 */
@Data @RequiredArgsConstructor
public class RenderedProblem {

  /**
   * Used for customizing the output of toString
   */
  public static class ToStringOptions {

    public ToStringOptions() {
    }

    public ToStringOptions(Consumer<ToStringOptions> customize) {
      if (customize != null) {
        customize.accept(this);
      }
    }

    /**
     * Applied after `prefixAll` and before `prefixChildren` `indentLevel` times.
     */
    public String indent = "  ";

    /**
     * All problem lines begin with this string
     */
    public String prefixAll = "";

    /**
     * All children have this applied once after any indenting
     */
    public String prefixChildren = "- ";

    /**
     * Problems are separated with this.  Use terminateWith if you want a final separator
     */
    public String separator = OsUtils.LINE_SEPARATOR;

    /**
     * Applied to the end of the string after all other text is output
     */
    public String terminateWith = "";
  }

  private static final ToStringOptions DEFAULT_TO_STRING_OPTIONS = new ToStringOptions();

  /**
   * The translated message for the source problem
   */
  public final String message;

  /**
   * The thing that caused all the problems
   */
  public final Problem source;

  /**
   * The number of ancestors this problem has
   */
  public final int indentLevel;

  /**
   * A list of RenderedProblems for each child in source
   */
  public final List<RenderedProblem> children;

  public RenderedProblem(String message, Problem source) {
    this(message, source, 0, Collections.emptyList());
  }

  /**
   * Convert this RenderedProblem in to a list, as per a depth-first search of this RenderedProblem
   * @return
   */
  public List<RenderedProblem> toList() {
    LinkedList<RenderedProblem> collected = new LinkedList<>();
    addAll(collected);
    return collected;
  }

  private void addAll(LinkedList<RenderedProblem> collected) {
    collected.add(this);

    for (RenderedProblem renderedProblem : children) {
      renderedProblem.addAll(collected);
    }
  }

  /**
   * Returns a line separated, indented version of this problem and its children
   */

  public String toString(Consumer<ToStringOptions> customize) {
    return toString(new ToStringOptions(customize));
  }

  public String toString(ToStringOptions options) {
    List<RenderedProblem> problems = toList();

    StringBuilder builder = new StringBuilder();
    for (RenderedProblem renderedProblem : problems) {
      if (builder.length() != 0) {
        builder.append(options.separator);
      }

      builder.append(options.prefixAll);
      if (renderedProblem.indentLevel > 0) {
        for (int i = 0; i < renderedProblem.indentLevel; i++) {
          builder.append(options.indent);
        }
        builder.append(options.prefixChildren);
      }
      builder.append(renderedProblem.message);
    }

    builder.append(options.terminateWith);

    return builder.toString();
  };

  /**
   * Returns a line separated, indented version of this problem and its children
   */
  @Override
  public String toString() {
    return toString(DEFAULT_TO_STRING_OPTIONS);
  };

}
