/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import java.util.List;
import java.util.Locale;
import java.util.Optional;

import nz.org.riskscape.engine.plugin.Plugin;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

/**
 * High level interface in to RiskScape's i18n subsystem.
 *
 */
public interface Messages {

  /**
   * @return the locale used by this Messages API if none is given.  In most contexts it should be safe to rely on this
   * instead of actually supplying one.  In a web-based future, we will probably proxy the messages subsystem per
   * request so that the locale is "correct".
   */
  default Locale getDefaultLocale() {
    return Locale.getDefault();
  }

  /**
   * @return the {@link MessageSource} used for getting object labels, e.g. nouns for things like projects, bookmarks
   * etc
   */
  MessageSource getLabels();

  /**
   * @return the {@link MessageSource} used for getting help text for various objects and concepts in RiskScape
   */
  MessageSource getHelp();

  /**
   * @return the {@link MessageSource} used for getting problem descriptions, e.g. "Hey user, you did this wrong thing"
   */
  MessageSource getProblems();

  /**
   * Used for cases where the help is very detailed and can span several
   * paragraphs. In cases like this the i18n message keys look like:
   *   [thing].description # brief
   *   [thing].description.1 # second paragraph
   *   [thing].description.2 # and so on
   * @return all paragraphs matching the object/parts in a list
   */
  List<String> getDetailedMessage(MessageSource messageSource, Locale locale, Object owner, String... parts);

  /**
   * @return a string containing the message from the given message source in the given locale, retrieved by
   * building keys based on metadata in the owner object, e.g. class name, object's id field
   */
  Optional<String> getMessage(MessageSource messageSource, Locale locale, Object owner, String... parts);

  /**
   * Same as {@link #getDetailedMessage(MessageSource, Locale, Object, String...)} but using default locale
   */
  default List<String> getDetailedMessage(MessageSource messageSource, Object owner, String... parts) {
    return getDetailedMessage(messageSource, getDefaultLocale(), owner, parts);
  }

  /**
   * Same as {@link #getMessage(MessageSource, Locale, Object, String...)} but using default locale
   */
  default Optional<String> getMessage(MessageSource messageSource, Object owner, String... parts) {
    return getMessage(messageSource, getDefaultLocale(), owner, parts);
  }

  /**
   * Convenience form for `getMessage(getHelp(), owner, parts)`, cos it's very commonly used
   */
  default Optional<String> getHelpMessage(Object owner, String... parts) {
    return getMessage(getHelp(), owner, parts);
  }

  /**
   * Convenience form for `getDetailedMessage(getHelp(), owner, parts)`, cos it's very commonly used
   */
  default List<String> getDetailedHelpMessage(Object owner, String... parts) {
    return getDetailedMessage(getHelp(), owner, parts);
  }

  /**
   * Construct a MessageSource from the named resource bundle base name.  i.e. newMessageSource("help") would create
   * you the same MessagesSource as getHelp().
   */
  MessageSource newMessageSource(String bundleName);

  /**
   * Turn a java object in to something that can be displayed as a simple, brief bit of text, serving as a plug-able,
   * translatable-toString mechanism for for RiskScape
   */
  String renderObject(Object object, Locale locale);

  /**
   * Turn a {@link Problem} in to a {@link RenderedProblem} - from there it can be transformed in to a string or
   * otherwise.  See TerminalCommand#renderProblem for an example.
   */
  RenderedProblem renderProblem(Problem problem, Locale locale);

  /**
   * Convenience form for `renderProblem(problem, Locale.getDefault())`
   */
  default RenderedProblem renderProblem(Problem problem) {
    return renderProblem(problem, getDefaultLocale());
  };

  /**
   * Convenience form for `renderProblem(Problems.caught(throwable), Locale.getDefault())`
   */
  default RenderedProblem renderProblem(Throwable throwable) {
    return renderProblem(Problems.caught(throwable), getDefaultLocale());
  };

  /**
   * Convenience form for `renderProblem(Problems.caught(throwable), locale)`
   */
  default RenderedProblem renderProblem(Throwable throwable, Locale locale) {
    return renderProblem(Problems.caught(throwable), locale);
  };

  /**
   * Populate the i18n subsystem with plugin resources from the given plugin
   */
  void addPluginResources(Plugin plugin);

}
