/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.geo;

import java.util.function.Function;

import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;

import lombok.RequiredArgsConstructor;

/**
 * A utility for automatically fixing invalid {@link Geometry}.
 */
@FunctionalInterface
public interface GeometryFixer {

  /**
   * A base implementation. Package private to allow for test access.
   */
  @RequiredArgsConstructor
  class Base implements GeometryFixer {

    private final Function<Geometry, Geometry> fixer;

    @Override
    public Geometry fix(Geometry geom) {
      GeometryFamily sourceFamily = GeometryFamily.from(geom);
      Geometry fixed = fixer.apply(geom);
      if (fixed.isEmpty()) {
        // the JTS Geometry fixer will always return a valid geometry, but to be valid it might just make
        // it empty. We would rather return the original geom in this case and know that its bad.
        return null;
      }
      if (fixed instanceof GeometryCollection) {
        // GeometryCollections may become hetrogenus (mixed families) during the fixing process. RiskScape
        // is adverse to hetrogenus geometry collections so we don't allow that.
        // TODO consider either we should drop the lower dimension parts and from the fixed geometry.
        GeometryCollection collection = (GeometryCollection) fixed;
        GeometryFamily family = null;
        for (int i = 0; i < collection.getNumGeometries(); i++) {
          Geometry part = collection.getGeometryN(i);
          if (family == null) {
            family = GeometryFamily.from(part);
          } else {
            if (!family.isSameFamily(part)) {
              return null;
            }
          }
        }
        if (collection.getNumGeometries() == 1) {
          // there's no point to a collection of one so let's unpack it. But only if the item is the
          // expected source family otherwise carry on
          Geometry onlyItem = collection.getGeometryN(0);
          if (sourceFamily.isSameFamily(onlyItem)) {
            return onlyItem;
          }
        }
      } else if (!sourceFamily.isSameFamily(fixed)) {
        // JTS Geometry fixer may change the geometry type (e.g a polygon could colapse to a line
        // or point). We don't want to allow that either.
        return null;
      }
      return fixed;
    }
  };

  /**
   * A {@link GeometryFixer} based on the JTS GeometryFixer.
   */
  GeometryFixer DEFAULT = new Base((geom) -> org.locationtech.jts.geom.util.GeometryFixer.fix(geom));

  /**
   * Attempt to fix geom (which ought to be invalid).
   *
   * @param geom geometry to fix
   * @return a fixed version of geom if fixing was possible, else null
   */
  Geometry fix(Geometry geom);

}
