/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.geo;

import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.Lineal;
import org.locationtech.jts.geom.Polygonal;
import org.locationtech.jts.geom.Puntal;

import lombok.Getter;

/**
 * An enumeration of the main Geometry type 'families' we support in RiskScape, ranked in terms of their
 * lowest-common-denominator-y-ness when intersecting each other, e.g. intersection of points and polygons results in
 * points, so polygons are ranked higher than points as they are more 'specific'.
 */
public enum GeometryFamily {
  PUNTAL(Puntal.class),
  LINEAL(Lineal.class),
  POLYGONAL(Polygonal.class);

  @Getter
  private Class<?> familyClass;

  /**
   * @return the GeometryFamily for the given geometry
   */
  public static GeometryFamily from(Geometry geometry) {
    return fromClass(geometry.getClass());
  }

  /**
   * Return the {@link GeometryFamily} of fromClass. If fromClass is not a specific geometry type
   * then null will be returned.
   *
   * @return the GeometryFamily for the given geometry class
   */
  public static GeometryFamily fromClass(Class fromClass) {
    for (GeometryFamily family : GeometryFamily.values()) {
      if (family.familyClass.isAssignableFrom(fromClass)) {
        return family;
      }
    }

    return null;
  }


  GeometryFamily(Class<?> clazz) {
    this.familyClass = clazz;
  }

  /**
   * Returns the lesser of the two geometry types - equivalent to the expected type when intersecting two geometries of
   * the given types, excluding weird peripheral geometries such as glancing points and lines
   */
  public GeometryFamily min(GeometryFamily rhs) {
    return this.ordinal() < rhs.ordinal() ? this : rhs;
  }

  /**
   * @return true if geom is a member of this geometry family, false otherwise
   */
  public boolean isSameFamily(Geometry geom) {
    return familyClass.isAssignableFrom(geom.getClass());
  }
}
