/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.agg;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizableFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.rl.agg.Accumulator;
import nz.org.riskscape.engine.rl.agg.AggregationFunction;
import nz.org.riskscape.engine.rl.agg.RealizedAggregateExpression;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.SimpleType;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;
import nz.org.riskscape.rl.ast.PropertyAccess;

/**
 * Adapts an {@link AggregationFunction} so that it can be called in a standard {@link RealizedExpression}.
 *
 * Supports only single argument functions, but could be extended to support more complex configurations
 */
@RequiredArgsConstructor
public class AggregationFunctionAdapter implements RiskscapeFunction, RealizableFunction {

  private final AggregationFunction aggfunction;

  @Override
  public Type getReturnType() {
    return aggfunction.getReturnType();
  }

  @Override
  public ArgumentList getArguments() {
    return aggfunction.getArguments();
  }

  @Override
  public List<Type> getArgumentTypes() {
    return getArguments().getArgumentTypes();
  }

  @Override
  public Object call(List<Object> args) {
    throw new UnsupportedOperationException();
  }

  @Override
  public Optional<AggregationFunction> getAggregationFunction() {
   return Optional.of(aggfunction);
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> argumentTypes) {

    if (argumentTypes.size() != getArgumentTypes().size()) {
      return ResultOrProblems.failed(ArgsProblems.get().wrongNumber(getArgumentTypes().size(), argumentTypes.size()));
    }

    return ProblemException.catching(() -> {
      RSList list = Types.findOrThrow(functionCall.getArguments().get(0), RSList.LIST_ANYTHING, argumentTypes.get(0));

      // NB get identifier name to be a literal of the expression for maybe better debugging?
      Struct kludgedInputType = Struct.of(SimpleType.DEFAULT_MEMBER_NAME, list.getContainedType());

      // The function args are copied except the first arg is replaced with our property access
      List<FunctionCall.Argument> kludgedFunctionCallArgs = new ArrayList<>(functionCall.getArguments());
      kludgedFunctionCallArgs.set(0, new FunctionCall.Argument(PropertyAccess.of(SimpleType.DEFAULT_MEMBER_NAME)));

      FunctionCall kludgedFunctionCall = new FunctionCall(
          functionCall.getIdentifier(),
          kludgedFunctionCallArgs
      );

      RealizedAggregateExpression resolvedSetFunction =
          aggfunction.realize(context, kludgedInputType, kludgedFunctionCall).
          getOrThrow();

      // return type is always nullable if a new accumulator is empty
      Type returnType = Nullable.ifTrue(
          resolvedSetFunction.newAccumulator().isEmpty(),
          resolvedSetFunction.getResultType());

      return new RiskscapeFunction() {

        @Override
        public Type getReturnType() {
          return returnType;
        }

        @Override
        public List<Type> getArgumentTypes() {
          return argumentTypes;
        }

        @Override
        public Object call(List<Object> args) {
          Accumulator accumulator = resolvedSetFunction.newAccumulator();
          List<?> values = (List<?>) args.get(0);
          Tuple value = new Tuple(kludgedInputType);
          for (Object object : values) {
            value.set(0, object);
            accumulator.accumulate(value);
          }

          if (accumulator.isEmpty()) {
            return null;
          } else {
            return accumulator.process();
          }
        }
      };
    });

  }

}
