/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import java.util.List;

import nz.org.riskscape.engine.types.FunctionType;

/**
 * A minimal {@link RiskscapeFunction} interface, which exists for two reasons: 1) store a reference
 * to a first-class function in a tuple 2) a minimal functional interface for use with creating minimal riskscape
 * functions without having to implement {@link RiskscapeFunction}.
 *
 * When an {@link UntypedFunction} is using for storing a reference to a higher order/first class function as part of a
 * RiskScape pipeline or expression, we don't want to carry around all the extra type metadata with each instance of the
 * function, as this is going to be static, and just means needlessly holding on to more bytes.  Hence this interface
 * exists, allowing implementations to be as light-touch on memory as possible.
 *
 * When stored in a tuple, an {@link UntypedFunction} is represented with a {@link FunctionType} type, which stores
 * the signature for all instances of the function that can be in a tuple of that type.
 *
 * Note that this interface used to extend {@link AutoCloseable}, but on reflection I think that is not going to work.
 * The 'functions-as-data' use case for {@link UntypedFunction} isn't going to support AutoCloseable, as tuples are not
 * closed, they are garbage collected.  The other use case for UntypedFunction is using it with
 * {@link RiskscapeFunction#create()}, but that now supports accepting an arbitrary list of {@link AutoCloseable}
 * objects to clean up.
 */
@FunctionalInterface
public interface UntypedFunction {

  /**
   * Execute the function with the given arguments.
   * @param args list of arguments to the function that should conform to the types in
   * {@link nz.org.riskscape.engine.function.ArgumentList#getArgumentTypes()}
   */
  Object call(List<Object> args);
}
