/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import java.util.List;
import java.util.stream.Collectors;

import lombok.AccessLevel;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Type;

/**
 * A {@link RiskscapeFunction} that wraps another one to short-circuit the target function to return null if any of the
 * args are null.
 */
@RequiredArgsConstructor(access = AccessLevel.PROTECTED)
public class NullSafeFunction implements RiskscapeFunction {

  /**
   * Wrap the given target function in a null-safe version. If any of the target function's arguments are not nullable,
   * then the resulting function will be a {@link NullSafeFunction} with those arguments that are not marked as nullable
   * being wrapped and applying the null-short-circuit behaviour when the function is called
   */
  public static RiskscapeFunction wrap(RiskscapeFunction target) {

    if (target instanceof NullSafeFunction) {
      return target;
    }

    if (target.getArgumentTypes().stream().allMatch(t -> t.find(Nullable.class).isPresent())) {
      return target;
    }

    List<Type> argumentTypes = target.getArgumentTypes().stream().map(t -> Nullable.of(t)).collect(Collectors.toList());
    Type returnType = Nullable.of(target.getReturnType());

    return new NullSafeFunction(target, argumentTypes, returnType, nullIndices(target));
  }

  /**
   * Wrap the given target function in a null-safe version.
   */
  public static RiskscapeFunction wrapIgnoringArgs(RiskscapeFunction target) {

    if (target instanceof NullSafeFunction) {
      return target;
    }

    List<Type> argumentTypes = target.getArgumentTypes().stream().map(t -> Nullable.of(t)).collect(Collectors.toList());
    Type returnType = Nullable.of(target.getReturnType());

    return new NullSafeFunction(target, argumentTypes, returnType, nullIndices(target));
  }

  private static boolean[] nullIndices(RiskscapeFunction target) {
    boolean[] indices = new boolean[target.getArgumentTypes().size()];
    for (int i = 0; i < indices.length; i++) {
      indices[i] = !target.getArgumentTypes().get(i).isNullable();
    }
    return indices;
  }

  @Getter
  private final RiskscapeFunction target;
  @Getter
  private final List<Type> argumentTypes;
  @Getter
  private final Type returnType;

  /**
   * for each argument, records true if the original receiver is not nullable
   */
  @Getter
  private final boolean[] notNullableIndices;

  @Override
  public Object call(List<Object> args) {
    for (int i = 0; i < notNullableIndices.length; i++) {
      if (notNullableIndices[i] && args.get(i) == null) {
        return null;
      }
    }

    return target.call(args);
  }

  @Override
  public void close() {
    target.close();
  }

  @Override
  public String toString() {
    return "NullSafe(" + target + ")";
  }


}
