/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import java.net.URI;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.google.common.collect.ImmutableMap;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Identified;
import nz.org.riskscape.engine.resource.Resource;
import nz.org.riskscape.engine.rl.RealizableFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.rl.agg.AggregationFunction;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * A {@link RiskscapeFunction} that is identified by its {@link #getId() }.
 *
 * Being identified by the ID is required for the function to be called from other contexts.
 */
public interface IdentifiedFunction extends RiskscapeFunction, Identified {

  @RequiredArgsConstructor
  class Wrapping implements IdentifiedFunction {

    protected final RiskscapeFunction wrapped;
    @Getter
    private final String id;
    @Getter
    private final String description;
    @Getter
    private final URI sourceURI;
    @Getter
    private final Category category;

    @Override
    public Object call(List<Object> args) {
      return wrapped.call(args);
    }

    @Override
    public List<Type> getArgumentTypes() {
      return wrapped.getArgumentTypes();
    }

    @Override
    public ArgumentList getArguments() {
      return wrapped.getArguments();
    };

    @Override
    public Type getReturnType() {
      return wrapped.getReturnType();
    }

    @Override
    public Optional<RealizableFunction> getRealizable() {
      return wrapped.getRealizable();
    }

    @Override
    public Optional<OverloadedFunction> getOverloaded() {
      return wrapped.getOverloaded();
    }

    @Override
    public Optional<AggregationFunction> getAggregationFunction() {
      return wrapped.getAggregationFunction();
    }

    @Override
    public ResultOrProblems<Boolean> validate(RealizationContext context) {
      return wrapped.validate(context);
    }

    @Override
    public String toString() {
      return String.format("Identified(id=%s %s)", id, wrapped);
    }

    @Override
    public void close() {
      wrapped.close();
    };

  }

  enum Category {
    GEOMETRY_PROCESSING,
    GEOMETRY_LOGICAL,
    MATHS,
    OPERATORS,
    LOGICAL,
    STRINGS,
    LANGUAGE,
    RISK_MODELLING,
    RISK_METRICS,
    MISC,
    UNASSIGNED
  }

  /**
   * @return a human readable description of the function
   */
  String getDescription();

  /**
   * @return a uri indicating the source of function. file/uri etc.
   */
  default URI getSourceURI() {
    return Resource.UNKNOWN_URI;
  }

  @Deprecated
  default String getSource() {
    return getSourceURI().toString();
  }

  /**
   * @return the category that this function belongs to
   */
  Category getCategory();

  /**
   * Get information about this function.
   */
  default String infoOutput() {
    return String.format("id: %s%ndescription: %s%nsource: %s%ninput types: %s%nreturn type: %s%n",
        getId(), getDescription(), getSource(), getArgumentTypes(), getReturnType());
  }

  /**
   * Returns a map-view of this fragility function
   */
  default Map<String, Object> toEntity() {
    return ImmutableMap.<String, Object>builder().
        put("id", getId()).
        put("description", getDescription()).
        put("argumentTypes", getArgumentTypes()).
        put("returnType", getReturnType()).
        build();
  }

  /**
   * @return true if this is a built-in function provided by RiskScape
   */
  default boolean isBuiltin() {
    return RiskscapeFunction.BUILT_IN.equals(getSourceURI());
  }
}
