/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import java.util.List;
import java.util.Map;

import nz.org.riskscape.engine.Identified;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.bind.BoundParameters;
import nz.org.riskscape.engine.bind.ParameterSet;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Defines a framework for building functions, e.g. Jython, Python, Classifier,
 * etc.
 *
 * Currently this only handles functions that are specified inline in the INI
 * file, e.g. `[function foo]`
 */
public interface FunctionFramework extends Identified {

  /**
   * The most neutral of priorities
   */
  int DEFAULT_PRIORITY = 0;

  /**
   * @return true if this is framework can build a function from the given INI definition.
   */
  boolean canBuild(Project project, Map<String, List<?>> unbound);

  /**
   * Returns a set of parameters that are used to define the function and its metadata.
   */
  ParameterSet getParameterSet();

  /**
   * Builds a function from an INI file section.
   * @param id The function's name
   * @param values The INI section's values, bound against {@link #getParameterSet()}.
   */
  ResultOrProblems<IdentifiedFunction> build(String id, Project project, BoundParameters values);

  /**
   * @return an integer value to use to sort FunctionFrameworks when multiple frameworks `canBuild` a function.
   * Functions are sorted in descending order, so that a framework with a higher priority value will be used in
   * preference to a framework with a lower priority value, i.e. priority 10 gets used ahead of priority 0.
   */
  default int getPriority() {
    return DEFAULT_PRIORITY;
  }

}
