/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import java.util.Arrays;
import java.util.List;
import java.util.function.BiPredicate;

import lombok.Getter;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.rl.TokenTypes;

/**
 * Convenience for adding {@link RiskscapeFunction} metadata to a java {@link BiPredicate}
 * function
 * @param <T> the type of the lhs of the predicate
 * @param <U> the type of the rhs of the predicate
 */
public class BinaryPredicateFunction<T, U> implements RiskscapeFunction {

  public static BinaryPredicateFunction<Object, Object> untyped(
      TokenTypes operation, Type lhs, Type rhs, BiPredicate<Object, Object> predicate) {
    return new BinaryPredicateFunction<>(operation, Object.class, lhs, Object.class, rhs, predicate);
  }

  public static <T> BinaryPredicateFunction<T, T> uniform(
      TokenTypes operation, Type type, Class<T> javaType, BiPredicate<T, T> predicate) {
    return new BinaryPredicateFunction<>(operation, javaType, type, javaType, type, predicate);
  }


  @Getter
  private TokenTypes operation;

  @Getter
  private final BiPredicate<T, U> predicate;
  @Getter
  private final Class<T> lhsJavaType;
  @Getter
  private final Class<U> rhsJavaType;
  @Getter
  private final List<Type> argumentTypes;
  @Getter
  private final Type lhsType;
  @Getter
  private final Type rhsType;

  @Getter
  private final Type returnType = Types.BOOLEAN;


  public BinaryPredicateFunction(TokenTypes operation,
      Class<T> lhsJavaType, Type lhsType, Class<U> rhsJavaType, Type rhsType,
      BiPredicate<T, U> predicate) {

    if (!lhsJavaType.isAssignableFrom(lhsType.internalType())) {
      throw new IllegalArgumentException("Can not construct a predicate function from mismatched java and riskscape "
          + "types");
    }

    if (!rhsJavaType.isAssignableFrom(rhsType.internalType())) {
      throw new IllegalArgumentException("Can not construct a predicate function from mismatched java and riskscape "
          + "types");
    }

    this.operation = operation;
    this.predicate = predicate;
    this.lhsJavaType= lhsJavaType;
    this.lhsType = lhsType;
    this.rhsJavaType = rhsJavaType;
    this.rhsType = rhsType;

    this.argumentTypes = Arrays.asList(lhsType, rhsType);
  }

  @Override
  public Object call(List<Object> args) {
    return predicate.test(lhsJavaType.cast(args.get(0)), rhsJavaType.cast(args.get(1)));
  }

  @Override
  public String toString() {
    return String.format("BinaryPredicate[operation=%s, types=%s]", operation, argumentTypes);
  }

}
