/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import java.util.Arrays;
import java.util.List;
import java.util.function.BinaryOperator;

import lombok.Getter;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.rl.TokenTypes;
import nz.org.riskscape.rl.ast.BinaryOperation;

/**
 * Convenience {@link RiskscapeFunction} for adding metadata to a java {@link BinaryOperator}
 * function, specifically meant for a riskscape language {@link BinaryOperation}
 */
public class BinaryOperatorFunction<T> implements RiskscapeFunction {

  @Getter
  private TokenTypes operation;

  @Getter
  private final BinaryOperator<T> operator;
  @Getter
  private final Class<T> javaType;
  @Getter
  private final Type returnType;
  @Getter
  private final List<Type> argumentTypes;

  public BinaryOperatorFunction(TokenTypes operation, BinaryOperator<T> operator, Class<T> javaType) {
    this(operation, operator, javaType, Types.fromJavaType(javaType));
  }

  /**
   * @param operation the corresponding lexing token for the operator
   * @param operator the function to call
   * @param javaType the java type that the operator accepts and returns
   * @param riskscapeType the riskscape type that the operator accepts and returns
   */
  public BinaryOperatorFunction(
      TokenTypes operation,
      BinaryOperator<T> operator,
      Class<T> javaType,
      Type riskscapeType
  ) {

    if (!riskscapeType.internalType().equals(javaType)) {
      throw new IllegalArgumentException("Can not construct an operator function from mismatched java and riskscape "
          + "types");
    }

    this.operation = operation;
    this.operator = operator;
    this.javaType = javaType;
    this.returnType = riskscapeType;
    this.argumentTypes = Arrays.asList(riskscapeType, riskscapeType);
  }

  @Override
  public Object call(List<Object> args) {
    return operator.apply(javaType.cast(args.get(0)), javaType.cast(args.get(1)));
  }

  @Override
  public String toString() {
    return String.format("BinaryOperator[operation=%s, type=%s]", operation, returnType);
  }

}
