/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import java.util.List;
import java.util.function.BiFunction;

import lombok.Getter;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;

public class BinaryFunction<T, U> implements RiskscapeFunction {

  @Getter
  private Type returnType;

  private BiFunction<T, U, Object> function;

  public BinaryFunction(
      FunctionArgument lhs,
      FunctionArgument rhs,
      Type returnType,
      BiFunction<T, U, Object> function
  ) {
    this.returnType = returnType;
    this.function = function;
    this.arguments = ArgumentList.fromArray(lhs, rhs);
    this.argumentTypes = arguments.getArgumentTypes();
  }

  public BinaryFunction(
      Type lhs,
      Type rhs,
      Type returnType,
      BiFunction<T, U, Object> function
  ) {
    this.returnType = returnType;
    this.function = function;
    this.arguments = ArgumentList.fromArray(
        new FunctionArgument("lhs", lhs),
        new FunctionArgument("rhs", rhs)
    );

    this.argumentTypes = arguments.getArgumentTypes();
  }

  public BinaryFunction(
      Class<T> lhsJavaType,
      Class<U> rhsJavaType,
      Type returnType,
      BiFunction<T, U, Object> function
  ) {
    this(Types.fromJavaType(lhsJavaType), Types.fromJavaType(rhsJavaType), returnType, function);
  }

  @Getter
  private final ArgumentList arguments;

  @Getter
  private final List<Type> argumentTypes;

  @Override
  @SuppressWarnings("unchecked")
  public Object call(List<Object> args) {
    T lhsArg = (T) args.get(0);
    U rhsArg = (U) args.get(1);

    return function.apply(lhsArg, rhsArg);
  }


}
