/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.data;

import java.util.Optional;

import nz.org.riskscape.engine.types.RelationType;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;

/**
 * An optional interface for an object that can describe itself in terms of RiskScape types.  This allows an arbitrary
 * instance of {@link SelfDescribingScalarData} to be injected in to a pipeline or an expression with correct type
 * information.
 */
public interface SelfDescribingScalarData {

  /**
   * Handy method that will attempt top return you a riskscape type for an object, first checking if it's self
   * describing, then falling back to {@link Types#fromJavaType(Class)}
   */
  static Optional<Type> getScalarDataType(Object instance) {
    if (instance instanceof SelfDescribingScalarData) {
      return Optional.of(((SelfDescribingScalarData) instance).getScalarDataType());
    } else {
      return Types.fromJavaTypeOptional(instance.getClass());
    }
  }

  /**
   * Returns the RiskScape type that describes this data when extracted from a project and added in to a pipeline as a
   * scalar value.
   *
   * It is referred to as the scalar type to make it clear it's not the same as the {@link Type} of the data contained
   * within it, e.g. the {@link Struct} type of the relation (non-scalar) vs {@link RelationType} that represents the
   * relation itself (scalar).
   */
  Type getScalarDataType();

}
