/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.data;

import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.geotools.api.coverage.Coverage;

import nz.org.riskscape.engine.Identified;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.ParameterSet;
import nz.org.riskscape.engine.relation.Relation;

/**
 * A {@link BookmarkResolver} can access the data that is being pointed to by a {@link Bookmark}.  It works with the
 * {@link ResolvedBookmark} interface to return data and error messages to the user.
 */
public interface BookmarkResolver extends Identified {

  /**
   * @param bookmark to resolve
   * @param context to bind any bookmark parameters with
   * @return a {@link ResolvedBookmark} capable of returning the data being referred to, or empty if this resolver can
   * not open the given bookmark.
   */
  Optional<ResolvedBookmark> resolve(Bookmark bookmark, BindingContext context) throws IOException;

  /**
   * @return a set of formats that allows a user to find out what sort of resources this resolver can
   * turn in to {@link Coverage}s or {@link Relation}s
   */
  Set<String> getFormats();

  /**
   * @return a list of file extensions that are supported by the given format.
   */
  List<String> getExtensions(String format);

  /**
   * @return a list of {@link Parameter}s that are accepted by this {@link BookmarkResolver} which configure
   * the resulting data that comes from the {@link ResolvedBookmark} that this resolver produces.
   * Each {@link Parameter}s name maps to an unparsed option contained by the {@link Bookmark#getUnparsed()} map.
   */
  ParameterSet getParameterSet();

  /**
   * @return an informative name for the resolver for users to identify it in debugging messages.
   */
  default String getName() {
    return getClass().getSimpleName();
  }

  @Override
  default String getId() {
    return getName();
  }
}
